from django.db import models
from django.core import validators


def validate_port_number(value):
    if not (1022 <= value <= 65535):
        raise validators.ValidationError("Port number must be between 1024 and 65535.")


class TvDetails(models.Model):
    """
    Represents details of TVs.

    This model stores information about TVs, including their name, description, unique ID,
    serial number, IP address, port number, secret key, and the groups they are associated with.

    Attributes:
        id (AutoField): Unique ID for the TV.
        name (CharField): Name of the TV.
        description (TextField): Description of the TV.
        unique_id (CharField): Unique ID of the TV (must be unique).
        serial_number (CharField): Serial number of the TV (must be unique).
        ipaddress (GenericIPAddressField): IP address of the TV.
        port_number (PositiveSmallIntegerField): Port number of the TV.
        secret (PasswordField): Secret key (password) for the TV.
    """

    id = models.AutoField(primary_key=True, help_text="Unique ID for the TV")
    name = models.CharField(max_length=100, help_text="Name of the TV", db_index=True, unique=True)
    description = models.TextField(help_text="Description of the TV", blank=True)
    unique_id = models.CharField(
        max_length=500, help_text="Unique ID of the TV", unique=True, null=True, blank=True
    )
    serial_number = models.CharField(
        max_length=500, help_text="Serial number of the TV", unique=True, null=True, blank=True
    )
    ipaddress = models.GenericIPAddressField(
        help_text="IP address of the TV", unique=True
    )
    port_number = models.PositiveSmallIntegerField(
        help_text="Port number of the TV", validators=[validate_port_number], default=5000
    )
    # TODO: Change this field to hasable to store key secretly
    secret = models.CharField(
        max_length=500, help_text="Secret key (password) for the TV", null=True, blank=True
    )

    def __str__(self) -> str:
        return self.name

    # def save(self, **args):
    #     base = super(TvDetails, self).save(**args)
    #     if not getattr(self, 'expected_status', None):
    #         self.expected_status = TVExpectedStatus.objects.create(tv_details=self)
    #     return base


class TVExpectedStatus(models.Model):
    """
    Expected status of TVs.

    This model represents the expected status of TVs, including details such as whether the TV is powered on, muted,
    subtitles enabled, subtitles language, and volume level. It is associated with a specific TV through a one-to-one
    relationship with the 'TvDetails' model.

    Attributes:
        tv_details (TvDetails): The associated TV details.
        channel_tmsid (str): The unique station key of the channel that is tuned in the TV.
        powered_on (bool): Indicates whether the TV should be powered on.
        muted (bool): Indicates whether the TV should be muted.
        subtitles_enabled (bool): Indicates whether subtitles should be enabled.
        subtitles_language (str): The language of the subtitles.
        volume (int): The volume level of the TV.
    """

    tv_details = models.OneToOneField(
        TvDetails,
        on_delete=models.CASCADE,
        related_name='expected_status',
        help_text="Associated TV details",
    )
    POWERED_ON_CHOICES = [(True, 'On'), (False, 'Off')]
    MUTED_CHOICES = [(True, 'Muted'), (False, 'Not Muted')]
    SUBTITLES_ENABLED_CHOICES = [(True, 'Enabled'), (False, 'Disabled')]
    CC_ENABLED_CHOICES = [(True, 'Enabled'), (False, 'Disabled')]

    channel_tmsid = models.CharField(
        max_length=100,
        help_text="channel name that is tuned in TV",
        null=True,
        default=None,
    )

    powered_on = models.BooleanField(
        choices=POWERED_ON_CHOICES,
        help_text="Should the TV Be powered on?",
        null=True,
        default=None,
    )
    muted = models.BooleanField(
        choices=MUTED_CHOICES,
        help_text="Should the TV Be muted?",
        null=True,
        default=None,
    )
    subtitles_enabled = models.BooleanField(
        choices=SUBTITLES_ENABLED_CHOICES,
        help_text="Should the subtitles Be enabled?",
        null=True,
        default=None,
    )
    subtitles_language = models.CharField(
        max_length=100,
        help_text="Subtitles language",
        null=True,
        default=None,
    )
    cc_enabled = models.BooleanField(
        choices=CC_ENABLED_CHOICES,
        help_text="Should the Closed Captions Be enabled?",
        null=True,
        default=None,
    )
    volume = models.PositiveSmallIntegerField(
        help_text="Volume level",
        null=True,
        default=None,
    )

    def __str__(self) -> str:
        return str(self.tv_details)
