from django.shortcuts import get_object_or_404
import hashlib
from rest_framework import viewsets
from rest_framework.response import Response
from rest_framework.decorators import action
from rest_framework.exceptions import NotFound, AuthenticationFailed
from .models import Sites, Group
from .serializers import SitesSerializer, GroupSerializer, SiteDaemonSerializer
from rest_framework.authentication import (
    TokenAuthentication,
    SessionAuthentication,
    BaseAuthentication,
)
from rest_framework.permissions import IsAuthenticated


class ClusterDaemonAuthentication(BaseAuthentication):
    """Athenticate against a HEADER set from environment variable"""

    def daemon_user(self):
        from django.contrib.auth import models

        user = models.User.objects.filter(
            username='daemon',
        ).first()
        if user is None:
            user = models.User.objects.create(
                username='daemon',
                email='support@atxnetworks.com',
            )
        return user

    def authenticate(self, request):
        import os

        header = request.headers.get('X-CLUSTER-AUTHENTICATION')
        local = os.environ.get('X_CLUSTER_AUTHENTICATION')
        if not local:
            raise AuthenticationFailed('Not configured for cluster operation')
        if not header:
            raise AuthenticationFailed('Not a cluster client')
        header_hash = hashlib.sha512(header.encode('utf-8')).hexdigest()
        local_hash = hashlib.sha512(local.encode('utf-8')).hexdigest()
        if header_hash != local_hash:
            raise AuthenticationFailed("Authentication failure for cluster auth")
        return (self.daemon_user(), None)


# Viewset for the Sites model
class SitesViewSet(viewsets.ModelViewSet):
    queryset = Sites.objects.all()
    serializer_class = SitesSerializer
    authentication_classes = [TokenAuthentication, SessionAuthentication]
    # permission_classes = [IsAuthenticated]

    """
    # TODO : remove the multi line comment post demo (After user authentication is implemented in UI) to allow groups for signed in users
    def get_queryset(self):
        user = self.request.user  # Get the user making the request
        # Filter the queryset to include only sites that the user is mapped to
        return Sites.objects.filter(usersite__user=user)
    """

    # TODO: restrict to use local-cluster secret
    @action(
        detail=True,
        methods=['get'],
        serializer_class=SiteDaemonSerializer,
        authentication_classes=[ClusterDaemonAuthentication],
    )
    def daemon_config(self, request, pk=None, format='.json'):
        """PRIVILEGED: Provide the configuration that the backend daemon will use to configure these devices

        Note: only cluster-side services can access this entry point

        returns a SiteDaemonSerializer structure
        """
        instance = get_object_or_404(Sites, id=pk)
        return Response(SiteDaemonSerializer(instance, many=False).data)


# Viewset for the Group model
class GroupViewSet(viewsets.ModelViewSet):
    queryset = Group.objects.all()
    serializer_class = GroupSerializer
    authentication_classes = [TokenAuthentication, SessionAuthentication]
    # permission_classes = [IsAuthenticated]

    def get_queryset(self):
        # Get the site ID from the URL parameters
        site_id = self.kwargs.get('site_id')

        # Check if site_id is provided in the URL
        if site_id is None:
            site_id = 1
            # TODO: Implement this post demo to mandate the site ID from the URL parameters else give out empty object
            # return Group.objects.none()  # Return an empty queryset

        # Check if the site with the specified site_id exists
        site = get_object_or_404(Sites, pk=site_id)

        # Filter the queryset to include only groups that belong to the specified site
        return Group.objects.filter(site_id=site_id)
