from django.urls import reverse
from rest_framework import status
from rest_framework.test import APITestCase
from .models import TvDetails, TVExpectedStatus
from .serializers import TVExpectedStatusSerializer
from django.test import TestCase

class TvDetailsViewSetTestCase(APITestCase):
    def setUp(self):
        # Create test data
        self.site_id = 1
        self.tv1 = TvDetails.objects.create(
            name="TV 1",
            description="TV 1 Description",
            unique_id="12345",
            serial_number="SN123",
            ipaddress="192.168.1.1",
            port_number=8080,
            secret="secret123",
        )
        self.tv2 = TvDetails.objects.create(
            name="TV 2",
            description="TV 2 Description",
            unique_id="67890",
            serial_number="SN456",
            ipaddress="192.168.1.2",
            port_number=8081,
            secret="secret456",
        )

    def test_retrieve_tv_details(self):
        url = reverse("tvdetails-detail", kwargs={"site_id": self.site_id, "pk": self.tv1.id})
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data["name"], "TV 1")

    def test_list_tv_details(self):
        url = reverse("tvdetails-list", kwargs={"site_id": self.site_id})
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(len(response.data), 2)  # Two TV details should exist

    def test_create_tv_details(self):
        url = reverse("tvdetails-list", kwargs={"site_id": self.site_id})
        data = {
            "name": "New TV",
            "description": "New TV Description",
            "unique_id": "54321",
            "serial_number": "SN789",
            "ipaddress": "192.168.1.3",
            "port_number": 8082,
            "secret": "secret789",
        }
        response = self.client.post(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(TvDetails.objects.count(), 3)  # There should be three TV details now

    def test_update_tv_details(self):
        url = reverse(
            "tvdetails-detail",
            kwargs={"site_id": self.site_id, "pk": self.tv1.id, "format": "json"},
        )
        data = {"name": "Updated TV Name", "description": "Updated TV Description"}
        response = self.client.patch(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            TvDetails.objects.get(id=self.tv1.id).name, "Updated TV Name"
        )

    def test_delete_tv_details(self):
        url = reverse("tvdetails-detail", kwargs={"site_id": self.site_id, "pk": self.tv2.id})
        response = self.client.delete(url)
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        self.assertEqual(TvDetails.objects.count(), 1)  # One TV detail should be deleted

    # Add more test cases as needed





class TVExpectedStatusTestCase(TestCase):
    def setUp(self):
        # Create a TVDetails instance for testing
        self.tv_details = TvDetails.objects.create(
            name="Test TV",
            description="Test TV Description",
            unique_id="12345",
            serial_number="SN123",
            ipaddress="192.168.1.1",
            port_number=8080,
            secret="secret123",
        )

    def test_create_tv_expected_status(self):
        tv_expected_status = TVExpectedStatus.objects.create(
            tv_details=self.tv_details,
            channel_tmsid="123456",
            powered_on=True,
            muted=False,
            subtitles_enabled=True,
            subtitles_language="English",
            cc_enabled=True,
            volume=50,
        )
        self.assertEqual(tv_expected_status.tv_details, self.tv_details)
        self.assertEqual(tv_expected_status.channel_tmsid, "123456")
        self.assertTrue(tv_expected_status.powered_on)
        self.assertFalse(tv_expected_status.muted)
        self.assertTrue(tv_expected_status.subtitles_enabled)
        self.assertEqual(tv_expected_status.subtitles_language, "English")
        self.assertTrue(tv_expected_status.cc_enabled)
        self.assertEqual(tv_expected_status.volume, 50)

    def test_update_tv_expected_status(self):
        tv_expected_status = TVExpectedStatus.objects.create(
            tv_details=self.tv_details,
            channel_tmsid="123456",
            powered_on=True,
            muted=False,
            subtitles_enabled=True,
            subtitles_language="English",
            cc_enabled=True,
            volume=50,
        )
        tv_expected_status.volume = 60
        tv_expected_status.save()
        updated_tv_expected_status = TVExpectedStatus.objects.get(id=tv_expected_status.id)
        self.assertEqual(updated_tv_expected_status.volume, 60)

    def test_delete_tv_expected_status(self):
        tv_expected_status = TVExpectedStatus.objects.create(
            tv_details=self.tv_details,
            channel_tmsid="123456",
            powered_on=True,
            muted=False,
            subtitles_enabled=True,
            subtitles_language="English",
            cc_enabled=True,
            volume=50,
        )
        tv_expected_status.delete()
        self.assertEqual(TVExpectedStatus.objects.count(), 0)

    def test_str_representation(self):
        tv_expected_status = TVExpectedStatus.objects.create(
            tv_details=self.tv_details,
            channel_tmsid="123456",
            powered_on=True,
            muted=False,
            subtitles_enabled=True,
            subtitles_language="English",
            cc_enabled=True,
            volume=50,
        )
        expected_str = str(self.tv_details)
        self.assertEqual(str(tv_expected_status), expected_str)

    def test_filter_by_tv_details(self):
        # Create multiple TVExpectedStatus objects for different TVs
        tv1 = TvDetails.objects.create(
            name="TV 1",
            description="TV 1 Description",
            unique_id="12345876",
            serial_number="SN12543",
            ipaddress="192.168.1.11",
            port_number=8080,
            secret="secret123",
        )
        tv2 = TvDetails.objects.create(
            name="TV 2",
            description="TV 2 Description",
            unique_id="678910",
            serial_number="SN45698",
            ipaddress="192.168.1.21",
            port_number=8081,
            secret="secret456",
        )
        TVExpectedStatus.objects.create(tv_details=tv1, channel_tmsid="123456")
        TVExpectedStatus.objects.create(tv_details=tv2, channel_tmsid="789012")
        
        # Filter TVExpectedStatus objects by TV details
        filtered_status = TVExpectedStatus.objects.filter(tv_details=tv1)
        
        self.assertEqual(filtered_status.count(), 1)
        self.assertEqual(filtered_status.first().channel_tmsid, "123456")
