from django.db import models
from django.core.exceptions import ValidationError
from sites.models import Sites, Group
from django.contrib.auth.models import User


class UserSite(models.Model):
    """
    Represents the relationship between users, sites, and groups within a system.

    This model defines a relationship between a user, a site, and an optional group, indicating whether the user is a site
    admin and/or a group admin for the given site and group.

    Attributes:
        user (ForeignKey): The user associated with this relationship.
        site (ForeignKey): The site associated with this relationship.
        group (ForeignKey, optional): The group associated with this relationship, if applicable.
        is_site_admin (BooleanField): Indicates whether the user is a site admin for the associated site.
        is_group_admin (BooleanField): Indicates whether the user is a group admin for the associated group, if applicable.
    """

    user = models.ForeignKey(User, on_delete=models.CASCADE, help_text="The user associated with this relationship.")
    site = models.ForeignKey(Sites, on_delete=models.CASCADE, help_text="The site associated with this relationship.")
    group = models.ForeignKey(Group, on_delete=models.CASCADE, null=True, blank=True,
                              help_text="The group associated with this relationship, if applicable.")
    is_site_admin = models.BooleanField(default=False,
                                        help_text="Indicates whether the user is a site admin for the associated site.")
    is_group_admin = models.BooleanField(default=False,
                                         help_text="Indicates whether the user is a group admin for the associated group, if applicable.")

    def __str__(self):
        """
        Returns a string representation of the UserSite object.
        Format: "{username} - {site_name} - {group_name if group else 'No Group'}"
        """
        return f"{self.user.username} - {self.site.name} - {self.group.name if self.group else 'No Group'}"

    def clean(self):
        """
        Custom validation method that checks whether the group belongs to the site.

        Raises:
            ValidationError: If the group does not belong to the site.
        """
        if self.group and self.group.site != self.site:
            raise ValidationError("The group must belong to the site.")

    def save(self, *args, **kwargs):
        """
        Overrides the save method to enforce the constraint: if is_site_admin is True, set is_group_admin to True.
        """
        if self.is_site_admin:
            self.is_group_admin = True
        super(UserSite, self).save(*args, **kwargs)
