from django.test import TestCase
from django.contrib.auth.models import User
from .models import UserSite, Sites, Group
from django.core.exceptions import ValidationError

class UserSiteModelTestCase(TestCase):
    def setUp(self):
        # Create a test user, site, and group
        self.user = User.objects.create_user(username='testuser', password='password')
        self.site = Sites.objects.create(name='Test Site')
        self.group = Group.objects.create(name='Test Group', site=self.site)

    def test_create_user_site_relationship(self):
        # Create a UserSite object
        user_site = UserSite.objects.create(
            user=self.user,
            site=self.site,
            group=self.group,
            is_site_admin=True,
            is_group_admin=False
        )

        # Check if the object was created successfully
        self.assertEqual(UserSite.objects.count(), 1)

        # Check the string representation of the object
        self.assertEqual(str(user_site), 'testuser - Test Site - Test Group')

    def test_invalid_group_relationship(self):
        # Attempt to create a UserSite object with an invalid group
        with self.assertRaises(ValidationError):
            UserSite.objects.create(
                user=self.user,
                site=self.site,
                group=Group.objects.create(name='Invalid Group', site=Sites.objects.create(name='Another Site')),
                is_site_admin=True,
                is_group_admin=False
            )

    def test_is_site_admin_constraint(self):
        # Create a UserSite object with is_site_admin set to True
        user_site = UserSite.objects.create(
            user=self.user,
            site=self.site,
            group=self.group,
            is_site_admin=True,
            is_group_admin=False
        )

        # Check if is_group_admin was automatically set to True
        self.assertTrue(user_site.is_group_admin)

    def test_update_user_site_relationship(self):
        # Create a UserSite object
        user_site = UserSite.objects.create(
            user=self.user,
            site=self.site,
            group=self.group,
            is_site_admin=False,
            is_group_admin=False
        )

        # Update the object
        user_site.is_site_admin = True
        user_site.save()

        # Check if is_group_admin was automatically set to True after the update
        self.assertTrue(user_site.is_group_admin)

    def test_delete_user_site_relationship(self):
        # Create a UserSite object
        user_site = UserSite.objects.create(
            user=self.user,
            site=self.site,
            group=self.group,
            is_site_admin=False,
            is_group_admin=False
        )

        # Delete the object
        user_site.delete()

        # Check that the object was deleted
        self.assertEqual(UserSite.objects.count(), 0)

    def test_default_values(self):
        # Create a UserSite object with no custom values
        user_site = UserSite.objects.create(
            user=self.user,
            site=self.site
        )

        # Check if default values are applied
        self.assertFalse(user_site.is_site_admin)
        self.assertFalse(user_site.is_group_admin)

    def test_clean_method_valid_group(self):
        # Create a UserSite object with a valid group
        user_site = UserSite(
            user=self.user,
            site=self.site,
            group=self.group,
            is_site_admin=True,
            is_group_admin=False
        )

        # This should pass validation
        user_site.clean()

    def test_clean_method_invalid_group(self):
        # Create a UserSite object with an invalid group
        invalid_group = Group.objects.create(name='Invalid Group', site=Sites.objects.create(name='Another Site'))
        user_site = UserSite(
            user=self.user,
            site=self.site,
            group=invalid_group,
            is_site_admin=True,
            is_group_admin=False
        )

        # This should raise a ValidationError
        with self.assertRaises(ValidationError):
            user_site.clean()

    def test_relationships(self):
        # Create a UserSite object
        user_site = UserSite.objects.create(
            user=self.user,
            site=self.site,
            group=self.group,
            is_site_admin=True,
            is_group_admin=False
        )

        # Check the relationships
        self.assertEqual(user_site.user, self.user)
        self.assertEqual(user_site.site, self.site)
        self.assertEqual(user_site.group, self.group)
