from django.urls import reverse
from rest_framework import status
from rest_framework.test import APITestCase
from django.contrib.auth.models import User
from .models import Sites, Group
from django.test import TestCase
from rest_framework.test import APIClient

class GroupViewSetTestCase(APITestCase):
    def setUp(self):
        self.user = User.objects.create_user(
            username="testuser", password="testpassword"
        )

        self.site = Sites.objects.create(
            name="Test Site", description="Test site description"
        )

        self.group1 = Group.objects.create(
            name="Group 1",
            description="Group 1 description",
            site=self.site,
        )

        self.group2 = Group.objects.create(
            name="Group 2",
            description="Group 2 description",
            site=self.site,
        )

    def test_list_groups(self):
        url = reverse("group-list", kwargs={"site_id": self.site.id})
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(len(response.data), 2)

    def test_retrieve_group(self):
        url = reverse(
            "group-detail",
            kwargs={"site_id": self.site.id, "pk": self.group1.id},
        )
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data["name"], "Group 1")

    def test_create_group(self):
        url = reverse("group-list", kwargs={"site_id": self.site.id})
        data = {
            "name": "Group 3",
            "description": "Group 3 description",
            "site": self.site.id,
        }
        self.client.force_authenticate(user=self.user)
        response = self.client.post(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Group.objects.count(), 3)

    def test_update_group(self):
        url = reverse(
            "group-detail",
            kwargs={"site_id": self.site.id, "pk": self.group1.id},
        )
        data = {"name": "Updated Group Name", "description": "Updated Group Description", "site": self.site.id}
        self.client.force_authenticate(user=self.user)
        response = self.client.put(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(Group.objects.get(id=self.group1.id).name, "Updated Group Name")


    def test_partial_update_group(self):
        url = reverse(
            "group-detail",
            kwargs={"site_id": self.site.id, "pk": self.group1.id},
        )
        data = {"description": "Updated Group Description"}
        self.client.force_authenticate(user=self.user)
        response = self.client.patch(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            Group.objects.get(id=self.group1.id).description, "Updated Group Description"
        )

    def test_delete_group(self):
        url = reverse(
            "group-detail",
            kwargs={"site_id": self.site.id, "pk": self.group1.id},
        )
        self.client.force_authenticate(user=self.user)
        response = self.client.delete(url)
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        self.assertFalse(Group.objects.filter(id=self.group1.id).exists())

    def test_delete_group_with_invalid_site_id(self):
        url = reverse(
            "group-detail",
            kwargs={"site_id": 999, "pk": self.group1.id},
        )
        self.client.force_authenticate(user=self.user)
        response = self.client.delete(url)
        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)

class SitesViewSetTestCase(TestCase):
    def setUp(self):
        self.client = APIClient()
        self.site1 = Sites.objects.create(name="Site 1", description="Description 1")
        self.site2 = Sites.objects.create(name="Site 2", description="Description 2")

    def test_list_sites(self):
        url = reverse("sites-list")
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(len(response.data), 2)  # Check if both sites are listed

    def test_retrieve_site(self):
        url = reverse("sites-detail", kwargs={"pk": self.site1.id})
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data["name"], "Site 1")

    def test_create_site(self):
        url = reverse("sites-list")
        data = {"name": "New Site", "description": "New Description", "epg_datasource": "EPG Data Source Value",
        "controller_details": "Controller Details Value"}
        response = self.client.post(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Sites.objects.count(), 3)  # Check if a new site was created

    def test_update_site(self):
        url = reverse("sites-detail", kwargs={"pk": self.site1.id})
        data = {"name": "Updated Site Name", "description": "Updated Site Description", "epg_datasource": "Updated EPG Data Source Value",
        "controller_details": "Updates Controller Details Value",}
        response = self.client.put(url, data, format="json")
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(Sites.objects.get(id=self.site1.id).name, "Updated Site Name")

    def test_delete_site(self):
        url = reverse("sites-detail", kwargs={"pk": self.site1.id})
        response = self.client.delete(url)
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        self.assertEqual(Sites.objects.count(), 1)  # Check if the site was deleted

    def test_create_site_missing_required_fields(self):
            url = reverse("sites-list")
            data = {"name": "New Site Name", "description": "New Site Description"}
            response = self.client.post(url, data, format="json")

            self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
            self.assertFalse(Sites.objects.filter(name="New Site Name").exists())
    
    def test_update_nonexistent_site(self):
        url = reverse("sites-detail", kwargs={"pk": 9999})  # Nonexistent site ID
        data = {
            "name": "Updated Site Name",
            "description": "Updated Site Description",
            "epg_datasource": "Updated EPG Data Source Value",
            "controller_details": "Updated Controller Details Value",
        }
        response = self.client.put(url, data, format="json")

        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)

    def test_delete_nonexistent_site(self):
        url = reverse("sites-detail", kwargs={"pk": 9999})  # Nonexistent site ID
        response = self.client.delete(url)

        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)
