from django.db import models
from django.contrib.postgres.fields import ArrayField

from django.db import models
from django.contrib.postgres.fields import ArrayField

from tv_screens.models import TvDetails, TVExpectedStatus
import logging

log = logging.getLogger(__name__)


class Sites(models.Model):
    """
    Represents a Site in the system.

    A Site is a location  with specific EPG data, controller details,
    and a list of favorite channels.

    Attributes:
        id (AutoField): Unique ID for the site.
        name (CharField): Name of the site.
        description (CharField): Description of the site.
        epg_datasource (CharField): EPG data source for the site.
        controller_details (CharField): Controller details for the site.
        favourite_channels (ArrayField of Integer): List of favorite channels for the site.
    """

    id = models.AutoField(primary_key=True, help_text="Unique ID for the site")
    name = models.CharField(max_length=100, help_text="Name of the site", db_index=True)
    description = models.TextField(help_text="Description of the site")
    epg_datasource = models.CharField(
        max_length=500,
        help_text="EPG data source for the site",
        default="",
        blank=True,
    )
    controller_details = models.CharField(
        max_length=500,
        help_text="Controller details for the site",
        default="",
        blank=True,
    )
    favourite_channels = ArrayField(
        models.CharField(null=True, blank=True),
        null=True,
        blank=True,
        help_text="List of favorite channels for the site",
    )

    def __str__(self) -> str:
        return self.name

    class Meta:
        verbose_name_plural = "Sites"

    def daemon_config(self):
        """Construct a configuration for use in the daemon mambda_daemon.models.BarConfig"""
        devices = (
            TvDetails.objects.filter(
                group__site=self,
            )
            .prefetch_related("expected_status")
            .order_by(
                "unique_id",
            )
            .all()
        )
        return {
            "site_id": self.id,
            "devices": [device.daemon_config() for device in devices],
            "epgfetch": self.epg_datasource,
        }

    def get_tvs(self):
        from tv_screens.models import TvDetails

        return TvDetails.objects.filter(groups__site=self).prefetch_related(
            "expected_status"
        )


class Group(models.Model):
    """
    Represents a Group associated with a Site.

    A Group can be a sub-group of another Group, allowing hierarchical organization.
    It contains a name, description, and a reference to the Site it belongs to.

    Attributes:
        id (AutoField): Unique ID for the group.
        name (CharField): Name of the group.
        parent_group (ForeignKey to 'self', optional): Parent group (if any) to which this group belongs.
        description (CharField): Description of the group.
        site (ForeignKey to Sites): Site to which the group belongs.
    """

    id = models.AutoField(primary_key=True, help_text="Unique ID for the group")
    name = models.CharField(
        max_length=100, help_text="Name of the group", db_index=True
    )
    parent_group = models.ForeignKey(
        "Group",
        null=True,
        blank=True,
        on_delete=models.CASCADE,
        related_name="child_groups",
        help_text="Parent group (if any)",
    )
    description = models.TextField(help_text="Description of the group")
    site = models.ForeignKey(
        Sites,
        on_delete=models.CASCADE,
        related_name="groups",
        help_text="Site to which the group belongs",
    )
    tvs = models.ManyToManyField(
        TvDetails,
        blank=True,
        related_name="groups",
        help_text="TVs associated with the group",
    )

    def __str__(self) -> str:
        return self.name

    class Meta:
        verbose_name_plural = "Groups"


def on_tv_change(instance, **kwargs):
    """There has been *some* change to some tv"""
    sites = Sites.objects.prefetch_related(
        "groups",
        "groups__tvs",
        "groups__tvs__expected_status",
    )
    if instance:
        sites = sites.filter(groups__tvs=instance)

    for site in sites.all():
        send_updated_config(site)


def on_tv_delete(instance, **kwargs):
    return on_tv_change(instance=None)


def on_tv_state_change(instance, **kwargs):
    """When a state changes, send updates"""
    return on_tv_change(instance=instance.tv_details)


def send_updated_config(site):
    """Send the entire updated configuration for the given site to the backend daemon

    Returns the requests.Response for the request to the daemon
    """
    from sites.serializers import SiteDaemonSerializer
    import requests
    from tv_screens.views import MAMBA_DAEMON_URL

    payload = SiteDaemonSerializer(instance=site).data
    # Make a POST request to the other API
    # Note: this is being done synchronously, so the calling
    # API will block until the daemon responds...
    log.warning("Sending config to the backend")
    response = requests.post(MAMBA_DAEMON_URL, json=payload)
    return response


models.signals.post_save.connect(on_tv_change, sender=TvDetails)
models.signals.post_save.connect(on_tv_state_change, sender=TVExpectedStatus)
models.signals.post_delete.connect(on_tv_delete, sender=TvDetails)
