# serializers.py
from rest_framework import serializers
from .models import TvDetails, TVExpectedStatus
from sites.models import Group, Sites


class GroupSerializer(serializers.ModelSerializer):
    class Meta:
        model = Group
        fields = ['id']


class TvDetailsSerializer(serializers.ModelSerializer):
    # groups = GroupSerializer(many=True)
    group_ids = serializers.SerializerMethodField()

    class Meta:
        model = TvDetails
        fields = [
            'id',
            'name',
            'description',
            'unique_id',
            'serial_number',
            'ipaddress',
            'port_number',
            'secret',
            'group_ids',
        ]

    def get_group_ids(self, obj):
        # Retrieve the IDs of the associated groups
        return obj.groups.values_list('id', flat=True)
    
    def create(self, validated_data):
        site_id = self.context.get('site_id')
        if not site_id:
            raise serializers.ValidationError("Site ID is required.")

        try:
            site = Sites.objects.get(id=site_id)
        except Sites.DoesNotExist:
            raise serializers.ValidationError("Site not found.")

        unique_id = validated_data.get('unique_id', '')
        validated_data['serial_number'] = unique_id.lower()
        tv = super().create(validated_data)

        default_group, created = Group.objects.get_or_create(
            site=site,
            name=f"Unassigned",
            defaults={"description": "Default group for the site"}
        )
        default_group.tvs.add(tv)
        
        return tv


class TVExpectedStatusSerializer(serializers.ModelSerializer):
    class Meta:
        model = TVExpectedStatus
        fields = '__all__'  # Include all fields from the model


class PassthroughField(serializers.Field):
    def to_representation(self, value):
        return value

class CovertToStirng(serializers.Field):
    def to_representation(self, value):
        return str(value)


class TvExpectedStatusDaemonSerializer(serializers.ModelSerializer):
    class Meta:
        model = TVExpectedStatus
        fields = ['power', 'mute', 'volume', 'station','subtitles','cc']

    power = PassthroughField(source='powered_on')
    mute = PassthroughField(source='muted')
    station = PassthroughField(source='channel_tmsid')
    #TODO: For some reason CC field is throwing error when we send boolean value to deamon So converting this to string now for demo later need to check this with deamon developer (Mike)
    cc = CovertToStirng(source='cc_enabled')
    subtitles = PassthroughField(source='subtitles_language')
    


class TvDetailsDaemonSerializer(serializers.ModelSerializer):
    class Meta:
        model = TvDetails
        fields = ['name', 'unique_key', 'ip_address', 'port', 'settings']

    unique_key = PassthroughField(source='unique_id')
    ip_address = PassthroughField(source='ipaddress')
    port = PassthroughField(source='port_number')
    settings = TvExpectedStatusDaemonSerializer(source='expected_status')


class SetTvConfigSerializer(serializers.Serializer):
    tv_id = serializers.ListField(child=serializers.IntegerField())
    channel_tmsid = serializers.CharField()
    station = serializers.CharField()
    powered_on = serializers.BooleanField()
    muted = serializers.BooleanField()
    subtitles_enabled = serializers.BooleanField()
    cc_enabled = serializers.BooleanField()
    subtitles_language = serializers.CharField()
    volume = serializers.IntegerField()


class IdentifySerializer(serializers.Serializer):
    tv_id = serializers.ListField(child=serializers.IntegerField())
