from django.http import HttpResponse
from django.shortcuts import render, get_object_or_404
from rest_framework import viewsets, status
from rest_framework.authentication import TokenAuthentication, SessionAuthentication
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView
from rest_framework.response import Response


from .models import TvDetails, TVExpectedStatus
from .serializers import (
    TvDetailsSerializer,
    TVExpectedStatusSerializer,
    SetTvConfigSerializer,
    IdentifySerializer,
)
from sites.models import Group

from barapplication.utilities.logger.source import get_logger
import requests

logger = get_logger(__name__)


MAMBA_DAEMON_URL = "http://mamba-daemon:8028/v1/config"
MAMBA_DAEMON_IDENTIFY_URL = "http://mamba-daemon:8028/v1/identify"


# Viewset for the TvDetails model
class TvDetailsViewSet(viewsets.ModelViewSet):
    queryset = TvDetails.objects.all()
    serializer_class = TvDetailsSerializer
    authentication_classes = [TokenAuthentication, SessionAuthentication]
    # permission_classes = [IsAuthenticated]
    
    def get_serializer_context(self):
        # Updating context to use the site id variable in serializer 
        context = super().get_serializer_context()
        context['site_id'] = self.kwargs.get('site_id')
        return context
    


# Viewset for the TV_ExpectedStatus model
class TV_ExpectedStatusViewSet(viewsets.ModelViewSet):
    queryset = TVExpectedStatus.objects.all()
    serializer_class = TVExpectedStatusSerializer
    authentication_classes = [TokenAuthentication, SessionAuthentication]
    # permission_classes = [IsAuthenticated]

    def get_queryset(self):
        # Get the TV ID from the URL parameter
        tv_id = self.kwargs.get('tv_id')
        site_id = self.kwargs.get('site_id')

        groups_for_tv = Group.objects.filter(tvs__id=tv_id)

        # Filter the groups to find those that belong to the specified site
        groups_in_site = groups_for_tv.filter(site_id=site_id)

        # If there are groups in the specified site, retrieve TVExpectedStatus objects
        if groups_in_site.exists():
            return TVExpectedStatus.objects.filter(tv_details__id=tv_id)
        else:
            # Return an empty queryset if the TV does not belong to any groups in the site
            return TVExpectedStatus.objects.none()


class SiteGroupsTVsExpectedStatus(APIView):
    def get(self, request, site_id):
        try:
            # Get all groups within the specified site
            groups = Group.objects.filter(site_id=site_id)

            all_tv_expected_status = []
            tv_details_set = set()  # Use a set to store unique TV details

            tv_data = []
            for group in groups:
                # Get all TVs within the group
                tvs = TvDetails.objects.filter(groups=group)

                for tv in tvs:
                    # Check if the TV details have already been processed
                    if tv in tv_details_set:
                        continue  # Skip this TV if its details have already been included

                    # Get the expected status of the TV
                    (
                        tv_expected_status,
                        created,
                    ) = TVExpectedStatus.objects.get_or_create(tv_details=tv)
                    tv_data.append(TVExpectedStatusSerializer(tv_expected_status).data)

                    # Add TV details to the set
                    tv_details_set.add(tv)

            all_tv_expected_status = tv_data
            return Response(all_tv_expected_status, status=status.HTTP_200_OK)
        except Exception as e:
            return Response(
                {'error': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR
            )


class SetTvConfig(APIView):
    def handle_config_request(self, request, site_id, partial):
        try:
            serializer = SetTvConfigSerializer(data=request.data, partial=partial)
            if not serializer.is_valid():
                return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

            from sites.models import (
                Sites,send_updated_config,
            )  # TODO: should be part of the api view here...

            site = get_object_or_404(
                Sites,
                id=site_id,
            )  # TODO: is this *your* site?
            tvs = (
                TvDetails.objects.filter(
                    id__in=serializer.validated_data['tv_id'],
                    groups__site=site,
                )
                .prefetch_related('groups', 'groups__site', 'expected_status')
                .all()
            )
            errors = []

            for tv in tvs:
                details = getattr(tv, 'expected_status', None)
                if details is None:
                    details = TVExpectedStatus.objects.create(
                        tv_details=tv,
                    )
                if 'volume' in serializer.validated_data and serializer.validated_data['volume'] > 0:
                    details.muted = False
                    request.data['muted'] = details.muted
                    
                if 'channel_tmsid' in serializer.validated_data:
                    details.powered_on = True
                    request.data['powered_on'] = details.powered_on
                serializer = TVExpectedStatusSerializer(
                    instance=details, data=request.data, partial=True
                )
                if serializer.is_valid():
                    serializer.save()
                else:
                    errors.extend(serializer.errors)
            if errors:
                return Response(
                    {
                        'errors': errors,
                    },
                    status=status.HTTP_400_BAD_REQUEST,
                )
            
            response = send_updated_config(site)

            # Check the response status code and return the result
            if response.status_code == 200:
                return Response(response.json(), status=status.HTTP_200_OK)
            else:
                logger.error(f"Response status code {response.status_code}")
                logger.error(f"Response error {response.json()}")
                return Response(response.json(), status=response.status_code)
        except TvDetails.DoesNotExist:
            return Response(
                {
                    'error': 'One or more TV IDs are invalid or do not belong to the specified site.'
                },
                status=status.HTTP_400_BAD_REQUEST,
            )
        except Exception as e:
            logger.exception("Failure in config setup")
            return Response(
                {'error': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def patch(self, request, site_id):
        return self.handle_config_request(request, site_id, partial=True)

    def post(self, request, site_id):
        return self.handle_config_request(request, site_id, partial=False)


class IdentifyTv(APIView):
    def post(self, request, site_id):
        try:
            serializer = IdentifySerializer(data=request.data)
            if not serializer.is_valid():
                return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

            message = request.data.get('message', None)
            tv_ids = request.data.get('tv_id', [])
            duration = request.data.get('duration', 10)

            # Validate TV IDs and map serial numbers
            if not tv_ids:
                return Response(
                    {'error': 'No TV IDs provided'}, status=status.HTTP_400_BAD_REQUEST
                )
            else:
                tv_id_to_serial = {}
                for tv_id in tv_ids:
                    try:
                        tvs = TvDetails.objects.filter(
                            id=tv_id, groups__site_id=site_id
                        )
                        if tvs.exists():
                            # Map the first TV's serial number to the TV ID
                            tv_id_to_serial[tv_id] = tvs.first().serial_number
                        else:
                            return Response(
                                {
                                    'error': f'TV ID {tv_id} is invalid or does not belong to the specified site.'
                                },
                                status=status.HTTP_400_BAD_REQUEST,
                            )
                    except TvDetails.DoesNotExist:
                        return Response(
                            {
                                'error': f'TV ID {tv_id} is invalid or does not belong to the specified site.'
                            },
                            status=status.HTTP_400_BAD_REQUEST,
                        )

                # Create payload with mapped serial numbers
                keys = [tv_id_to_serial[tv_id] for tv_id in tv_ids]

                payload = {"message": message, "keys": keys, "duration": duration}

                response = requests.post(MAMBA_DAEMON_IDENTIFY_URL, json=payload)

                # Check the response status code and return the result
                if response.status_code == 200:
                    return Response({"message": "Success"}, status=status.HTTP_200_OK)
                else:
                    return Response(response.json(), status=response.status_code)
        except Exception as e:
            return Response(
                {'error': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR
            )


def home(request):
    return render(request=request, template_name="tv_screens/home.html", context={})


def management(request):
    return render(
        request=request, template_name="tv_screens/management.html", context={}
    )


def tv_screen(request):
    return HttpResponse("Hello, world. You're at the tv_screens page.")



