"""Decorator to run an operation with a different euid/egid"""
import os
from functools import wraps
import pwd, grp
from atxstyle.sixish import unicode

class AsUser( object ):
    """decorator/context manager that switches euid/egid to provided values
    
    This allows you to, for instance, write a file or run an operation
    as a given user *while you are sudo*
    
    uid -- str:username or int:uid as which to run (None to skip)
    gid -- str:groupname or int:gid as which to run (None to skip)
    root_only -- if True, only attempt to set up uid/gid when we are eguid == 0
    sudo_user -- if True, use os.environ('SUDO_USER') as uid and gid if they are null
    
    Usage::
    
        with AsUser( sudo_user=True ):
            twrite.twrite('file-that-product-user-should-own', 'some content')
    """
    def __init__( self, uid=None, gid=None, root_only=True, sudo_user=False ):
        if not uid and sudo_user:
            uid = os.environ.get('SUDO_USER')
        if not gid and sudo_user:
            gid = os.environ.get('SUDO_USER')
        if isinstance( uid, (bytes,unicode)):
            try:
                uid = pwd.getpwnam( uid ).pw_uid
            except KeyError:
                uid = None
        self.uid = uid 
        if isinstance( gid, (bytes,unicode)):
            try:
                gid = grp.getgrnam( gid ).gr_gid
            except KeyError:
                gid = None
        self.gid = gid
        self.root_only = root_only
        self.previous_uid = None
        self.previous_gid = None
    def __enter__( self, *args ):
        if (not self.root_only) or os.getegid() == 0:
            if self.gid is not None:
                self.previous_gid = os.getegid()
                os.setegid( self.gid )
            if self.uid is not None:
                self.previous_uid = os.geteuid()
                os.seteuid( self.uid )
    def __exit__( self, *args ):
        if self.previous_uid is not None:
            os.seteuid( self.previous_uid )
        if self.previous_gid is not None:
            os.setegid( self.previous_gid )
    def __call__(self,function):
        """Decorator version of the function"""
        @wraps(function)
        def with_asuser( *args, **named ):
            with self:
                return function( *args, **named )
        return with_asuser

if __name__ == "__main__":
    # this test suite obviously requires sudo to run...
    from fussy import twrite
    with AsUser( sudo_user=True ):
        fname = '/tmp/moo'
        twrite.twrite( fname, 'example' )
    stat = os.stat( fname )
    if os.geteuid() == 0:
        assert stat.st_uid == pwd.getpwnam( os.environ.get('SUDO_USER') ).pw_uid, stat.st_uid
    os.remove( '/tmp/moo' )
    twrite.twrite( '/tmp/moo-outside', 'example' )
    stat = os.stat( '/tmp/moo-outside' )
    assert stat.st_uid == os.geteuid(), stat.st_uid
    os.remove('/tmp/moo-outside' )
