"""Asyncio operations for generic file/process comms"""
import asyncio, time, logging, os, json
from functools import wraps
import aiofiles

log = logging.getLogger(__name__)


class ProcessError(RuntimeError):
    @property
    def returncode(self):
        return self.args[0]

    @property
    def command(self):
        return self.args[1]

    @property
    def stdout(self):
        return self.args[2]

    @property
    def stderr(self):
        return self.args[3]


def timestamp():
    return time.time()


def with_async_lock(function):
    lock = asyncio.Lock()

    @wraps(function)
    async def with_async_lock(*args, **named):
        await lock.acquire()
        try:
            return await function(*args, **named)
        finally:
            lock.release()

    return with_async_lock


def with_async_log(function):
    log = logging.getLogger(function.__module__)

    @wraps(function)
    async def with_async_log(*args, **named):
        try:
            return await function(*args, **named)
        except Exception as err:
            log.exception(
                'Error in %s(*%s,**%s)', function.__name__, repr(args), repr(named)
            )
            raise

    return with_async_log


async def run_process(command, input=None):
    """Run a process, wait for it to return"""
    if isinstance(command, list):
        # log.debug('> %s', command if isinstance(command, str) else ' '.join(command))
        proc = await asyncio.create_subprocess_exec(
            *command,
            stdin=asyncio.subprocess.PIPE if input else None,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
    else:
        # log.debug('S> %s', command if isinstance(command, str) else ' '.join(command))
        proc = await asyncio.create_subprocess_shell(
            command,
            stdin=asyncio.subprocess.PIPE if input else None,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
    try:
        output, err = await proc.communicate(input)
    except asyncio.CancelledError as err:
        if proc.returncode is None:
            proc.kill()
        raise
    if proc.returncode:
        log.warning(
            'Error on %s:\n%s',
            " ".join(command) if isinstance(command, list) else command,
            err.decode('utf-8'),
        )
        raise ProcessError(proc.returncode, command, output, err)
    else:
        # if somehow communicate exited without finishing...
        if proc.returncode is None:
            proc.kill()
    return output, err


async def running_pids(program):
    """Run pgrep to find given program"""
    try:
        output, err = await run_process(['pgrep', '-f', program])
    except RuntimeError as err:
        # 1 being "no matches" for pgrep
        if err.args[0] == 1:
            return []
        raise
    output = output.decode('utf-8').strip().split()
    if output:
        return [int(x) for x in output]
    return []


async def atwrite(filename, content):
    """Asynchronous transactional write to filename"""
    target = filename + '~'
    if isinstance(content, str):
        content = content.encode('utf-8')
    log.info("Writing file: %s with %s bytes", target, len(content))
    async with aiofiles.open(target, 'wb') as fh:
        await fh.write(content)
    log.info("Renaming %s to %s", target, filename)
    os.rename(target, filename)


async def aread(filename, mode='rb'):
    """Asynchronously read the contents of the whole file into memory"""
    async with aiofiles.open(filename, mode) as fh:
        content = await fh.read()
    return content


async def aread_json(name):
    """Read filename, load with json.loads"""
    try:
        contents = await aread(name)
    except Exception as e:
        log.error('Error reading %s: %s', name, e)
        return

    try:
        return json.loads(contents)
    except Exception as e:
        log.error('Error loading %s as JSON: %s', name, e)
        return


async def wait_file(filename, poll=0.25):
    """Wait for a file to exist"""
    while True:
        if os.path.exists(filename):
            return True
        await asyncio.sleep(poll)
