"""Provides utility methods to ensure that multiple authe backends can fuction 
together without issue"""

from django.utils.safestring import mark_safe
from django.conf import settings

class AuthBackendManager(object):
    standard_error_message = mark_safe(""
        "Cannot enable this authentication scheme "
        "while another is enabled.<br>"
        "Please ensure that all other authentication "
        "schemes are disabled and try again.")

    @classmethod
    def _is_backend_installed(cls, backend_name, 
                              backends=settings.AUTHENTICATION_BACKENDS):
        """Determine if backend specified is in Django's AUTHENTICATION_BACKENDS."""
        for backend_str in backends:
            if backend_name in backend_str:
                return True
        return False

    @classmethod
    def get_enabled_backend(cls):
        if cls._is_backend_installed('ldap'):
            from ldapauth.models import LdapConfig
            if LdapConfig.is_backend_enabled():
                return 'ldap'

        if cls._is_backend_installed('tacacs'):
            from tacacsauth.models import TacacsConfig
            if TacacsConfig.is_backend_enabled():
                return 'tacacs'

        return None

    @classmethod
    def can_enable(cls, backend_name):
        """Determine if an auth. backend can be enabled. Returns True/False

        As per business rules, we do not want multiple auth. backends to be enabled
        at the same time. Backends will call this function when a user requests a
        backend to be enabled. This function will then check to ensure that no other
        backend is enabled and will then return True or False accordingly.
        """
        enabled_backend = cls.get_enabled_backend()
        if enabled_backend and enabled_backend != backend_name:
            return False
        return True
