"""Implements locking for azure filesystems which don't have flock

Moves the lock into the redis server rather than using the filesystem at all...
"""
import os, hashlib, socket, time, logging, uuid
import traceback as traceback_module
from atxstyle import redissend
import psutil
from fussy import cronlock

log = logging.getLogger(__name__)
# log.setLevel(logging.INFO)


class AzureFSLock(object):
    """Loose cronlock-like thing that doesn't rely on flock, which doesn't work on azure filesystems"""

    def __init__(self, filename, timeout=30):
        self.filename = filename
        self.timeout = timeout
        self.key = hashlib.md5(self.filename.encode('utf-8')).hexdigest()
        self.lock_marker = str(uuid.uuid4())

    @property
    def final_key(self):
        return 'lock.azurefs.%s' % (self.key,)

    def __enter__(self):
        try:
            self.channel = redissend.get_channel()

            final_key = self.final_key
            until = self.until = time.time() + self.timeout
            while time.time() < until:
                if not self.channel.set(
                    # nx here is saying we'll only set if the key is not already set,
                    # with "how many set" being the return value...
                    final_key,
                    self.lock_marker,
                    nx=True,
                    ex=self.timeout * 20,
                ):
                    log.info("Contention on lock %s waiting", self.final_key)
                    time.sleep(0.5)
                else:
                    return True
        except Exception as err:
            log.exception('Failure during azurefs enter')
        raise cronlock.Busy('Contention on the lock')

    def __exit__(self, exc_type, exc_value, traceback):
        self.channel = redissend.get_channel()

        final_key = self.final_key
        if exc_type:
            log.warning(
                "Exiting on error for AzureFSLock: %s",
                traceback_module.format_tb(traceback),
            )
        # NOTE: if our process has taken longer than 30 seconds then the
        # lock might be replaced by something else...
        if not self.channel.delete(self.final_key):
            log.info("Our key timed out before we hit __exit__")
