"""basic fab-file features used in all projects

use::

    fab --list

to get a list of the available commands within a product directory.
"""
from __future__ import print_function
import distutils, logging

try:
    from lzma import FILTER_ARMTHUMB
except ImportError as err:  # python 2.7 doesn't have this...
    pass
from atxstyle import fabmetadata

logging.getLogger('paramiko.transport').setLevel(logging.WARNING)
logging.basicConfig(level=logging.INFO)
log = logging.getLogger(__name__)
from atxstyle.sixish import unicode, as_unicode, as_bytes
from osupdates import ostarget
from osupdates.decorators import tempdir

try:
    from urllib import parse
except ImportError:
    from urlparse import urlparse as parse
from fabric.api import (
    sudo,
    run,
    env,
    settings,
    local,
    lcd,
    put,
    get,
    cd,
    parallel,
)

env['sudo_prefix'] += '-E '
from fabric import state

state.output.exceptions = True
assert state
from fabric.contrib import project
from fabric.contrib.files import exists
from fussy import nbio
import atxstyle
from atxstyle.envvar import env_var

assert atxstyle
import os, subprocess, glob, datetime, time, re, json, shutil
from six.moves import cStringIO as StringIO
from contextlib import contextmanager
from atxstyle.fabmetadata import *
from . import fabmetadata

# env.sudo_prefix = "sudo -H -S -p '%(sudo_prompt)s' "
HERE = os.path.dirname(__file__)
SOURCE_ROOT = os.path.abspath(os.path.join(HERE, '..', '..'))
COMMON_CONFIG = os.path.abspath(os.path.join(HERE, '..', 'common-config'))
RED5_APPS = os.path.join(SOURCE_ROOT, 'flashpreview', 'usr')
VERSATIVE_INST = os.path.join(SOURCE_ROOT, 'versativeinst')
SIDELOADS = os.path.join(SOURCE_ROOT, 'sideloads')
DEFAULT_SKU = env_var('DEFAULT_SKU', None)
INSTALL_POSTGIS = os.path.join(
    SOURCE_ROOT, 'configserver', 'create_template_postgis-debian.sh'
)
SSH_KEY = os.path.join(
    SOURCE_ROOT, 'atxstyle', 'common-config', 'home', '.ssh', 'authorized_keys'
)
PATCHES = os.path.abspath(os.path.join(HERE, '..', '..', 'patches'))
SELINUX_POLICIES = os.path.abspath(os.path.join(HERE, '..', '..', 'selinux-policies'))
env.default_js_variant = 'prod'
env.abort_exception = RuntimeError

TARGET_PYTHON = env.target_python = env_var('TARGET_PYTHON', 'python3.6')
BAMBOO_VERSION = ''
if 'BAMBOO_VERSION' in os.environ:
    BAMBOO_VERSION = os.environ['BAMBOO_VERSION']

DEV_INSTALLS = []
env.DEV_UPSTREAMS = DEV_UPSTREAMS = [
    #    ('fussy','bzr branch lp:fussy'),
    #    ('globalsub','bzr branch lp:globalsub'),
    #    ('mockproc','bzr branch lp:mockproc'),
    #    ('presentationviewer','bzr branch lp:presentationviewer'),
    #    ('mcastsocket','git clone https://github.com/mcfletch/mcastsocket.git'),
]

env.target_os = ostarget.target_os_by_codename(
    env_var('TARGET_OS', 'centos7'),
    remote=True,
)
REVERSE_LINKS = env.target_os.name == 'petalinux'

if TARGET_PYTHON == 'python2.7':
    GET_PIP = os.path.join(HERE, '..', 'get-pip-python2.7.py')
else:
    GET_PIP = os.path.join(HERE, '..', 'get-pip.py')


@traced
def ospath_func(func, filename):
    """Run a python os.path function on filename"""
    return run(
        '''python -c "import os, sys; print(os.path.%(func)s(sys.argv[1]))" '%(filename)s' '''
        % locals()
    ).strip()


# Don't use the fabric versions, as they suck rocks through straws...
def islink(link):
    return ospath_func('islink', link) == 'True'


def exists(filename):
    return ospath_func('exists', filename) == 'True'


def rsync_project(*args, **named):
    """project.rsync_project but first fixes/checks authorized keys before attempt"""
    user = env.user
    group = env.group
    sudo('chown -R %(user)s:%(group)s /home/%(user)s/.ssh' % locals())
    sudo('chmod -R go-rwx /home/%(user)s/.ssh' % locals())
    return project.rsync_project(*args, **named)


def load_requirements(reqfile, base=False, python_version=TARGET_PYTHON):
    """Load requirements (from `reqs.py`) with default of target python version"""
    return ostarget.load_requirements(reqfile, base, python_version)


load_dependencies = ostarget.load_dependencies


def guess_target_os():
    """Look at the machine to determine the OS release/version"""
    if exists('/etc/lsb-release'):
        codename = [
            line[1]
            for line in [
                line.split('=', 1) for line in run('cat /etc/lsb-release').splitlines()
            ]
            if line[0] == 'DISTRIB_CODENAME'
        ][0]
        env.target_os = ostarget.target_os_by_codename(codename)
        return env.target_os
    elif exists('/etc/centos-release'):
        base = 'centos'
        content = run('cat /etc/lsb-release')
        vmatch = re.compile('release (?P<version>[0-9.]+) ')
        version = vmatch.search(content).group('version')
        codename = base + (version.split('.')[0])
        env.target_os = ostarget.target_os_by_codename(codename)
        return env.target_os
    elif exists('/etc/petalinux'):
        base = 'petalinux'
        env.target_os = ostarget.target_os_by_codename(base)
        return env.target_os
    raise RuntimeError("Does not seem to be a CentOS or Ubuntu machine")


@traced
def manual_ntp():
    env.target_os.manual_ntp()


@traced
def ensure_dependencies(sku=None):
    """Build: make sure the dependencies for the product are installed"""
    # we need rsync to sync over the rest of our bits and bobs
    env.target_os.dependencies(['rsync'])
    common_etc(
        root='',
        firmware_embed_only=False,
        directories=[
            ('etc/yum.repos.d', 'etc/yum.repos.d'),
            ('etc/pki', 'etc/pki'),
        ],
        remote=True,
    )
    env.target_os.repo_update()
    dependencies = None
    if getattr(env, 'DEPENDENCIES', None):
        dependencies = env.DEPENDENCIES
    else:
        dependencies = env.target_os.deps
    if TARGET_PYTHON != 'python27':
        if hasattr(env.target_os, 'remove_python27'):
            env.target_os.remove_python27()

    env.target_os.dependencies(dependencies=dependencies)

    # everything needs these three
    def have_service(x):
        for depset in dependencies:
            if x in depset:
                return True
        return False

    for service in ['haveged', 'supervisor', 'nginx']:
        if have_service(service):
            env.target_os.enable_service(service)
            env.target_os.restart_service(service, allow_failures=True)


def use_apt_cache(ip='192.168.15.22'):
    env.target_os.sudo(
        '''echo 'Acquire::http { Proxy "http://%(ip)s:3142"; };' > /etc/apt/apt.conf.d/01proxy'''
        % locals()
    )


def ubuntu_kernel_downgrade():
    if env.target_os.name == 'precise':
        import re

        current = run('uname -r')
        version = [int(x) for x in re.compile('\d+').findall(current)]
        if version[:2] == [3, 13] and (not os.environ.get('BUILD')):
            sudo(
                'apt-get install -y linux-image-3.2.0-126-generic linux-headers-3.2.0-126 linux-headers-3.2.0-126-generic linux-image-3.2.0-126-generic'
            )
            with settings(warn_only=True):
                sudo(
                    'apt-get purge -y linux-generic-lts-trusty linux-headers-generic-lts-trusty linux-image-generic-lts-trusty'
                )
                sudo('rm /boot/*3.13*')
            env.target_os.update_grub()


def ubuntu_kernel_saucy():
    """Install obsolete/unsupported kernel from saucy which supports our drivers *and* the mini hardware"""
    if env.target_os.name == 'precise':
        current = run('uname -r')
        if '3.11.0' not in current:
            sudo(
                'apt-get install -y --install-recommends linux-image-generic-lts-saucy linux-headers-generic-lts-saucy linux-generic-lts-saucy'
            )
            with settings(warn_only=True):
                sudo(
                    'apt-get purge -y linux-generic-lts-trusty linux-headers-generic-lts-trusty linux-image-generic-lts-trusty'
                )
                sudo('rm /boot/*3.13*')
            env.target_os.update_grub()


# def centos_selinux_nginx_sockets():
#    source = os.path.join(SELINUX_POLICIES, 'nginx.pp')
#    target = '/home/%s/nginx.pp' % env.user
#    put( source, target )
#    sudo( 'semodule -i %s' % target )
#    sudo( 'systemctl daemon-reload' )


def random_password():
    import secrets as random, string

    chars = string.letters + string.digits
    return ''.join([random.choice(chars) for i in range(10)])


def restart_service(service):
    env.target_os.restart_service(service)


@traced
def create_product_user(user=None, sudo_user='atxproduct', hashed_password=None):
    """Create the expected administrative user (connects as user sudo_user)

    Creates the product user and adds them to adm and sudo groups
    """
    user = user or env.user
    with settings(user=sudo_user, warn_only=True):
        env.target_os.add_user(
            user, ('adm', 'sudo', 'dialout', 'plugdev', 'wheel', 'frontend', 'www-data')
        )
        env.target_os.ensure_lines_in_file(
            '/etc/sudoers',
            required=['%wheel        ALL=(ALL)       ALL'],
            disallowed=[re.compile(r'^#[ ]*%wheel[ \t]*ALL[ \t]*')],
        )
        # add_user already asks you for the password...
        # sudo( 'passwd %(user)s'%locals())
        install_ssh_keys_su(user)
        if hashed_password:
            env.target_os.set_hashed_password(user, hashed_password)
        else:
            sudo('passwd %(user)s' % locals())


@traced
def use_bash_shell(user=None):
    user = user or env.user
    sudo('chsh -s /bin/bash %(user)s' % locals())


def our_project(name):
    """Is this project one of ours?"""
    if name.startswith('django.') or name.startswith('django_'):
        return None
    try:
        path = os.path.abspath(os.path.normpath(find_dist(name)))
    except (ImportError, RuntimeError):
        return None
    if not '..' in os.path.relpath(path, SOURCE_ROOT):
        return path
    return None


def projects_from_settings(installed_apps=None):
    if installed_apps is None:
        installed_apps = product_settings().INSTALLED_APPS
    base = [
        path
        for path in [our_project(name.split('.')[0]) for name in installed_apps]
        if path
    ]
    # always need these for VADA...
    base.append(find_dist('osupdates'))
    if 'encprofile' in installed_apps:
        base.extend(
            [
                find_dist(name)
                for name in [
                    'scheduler',
                    'parseeas',
                    'pullsched',
                    'snmpagents',
                    'epgclient',
                    'epgfetch',
                    'tsstructure',
                    'videodefs',
                    'reactforms',
                    'restore',
                ]
                if name not in installed_apps
            ]
        )
        if env.target_os.name.startswith('centos'):
            base.extend(
                [
                    'vadagst',
                ]
            )
        if None in base:
            raise RuntimeError(
                "Seem to be missing packages in local/dev installation environment"
            )
    return base


# OS-level updates
@traced
def install_ssh_keys(force=False):
    if os.path.exists(SSH_KEY):
        # we have a local RSA key, make it available as authorized_keys
        with settings(warn_only=True):
            if not exists('~%s/.ssh' % (env.user,)):
                run('mkdir -p ~%s/.ssh' % (env.user,))
                run('chmod 0700 ~%s/.ssh' % (env.user,))
            if force or not exists('~%s/.ssh/authorized_keys' % (env.user,)):
                # only if we do *not* currently have an authorized keys!
                content = open(SSH_KEY).read().strip()
                local = SSH_KEY + '.local'
                if os.path.exists(local):
                    content = "\n".join([content, open(local).read().strip(), ''])
                put(
                    StringIO(content),
                    '/home/%s/.ssh/authorized_keys' % (env.user,),
                    mode=0o644,
                )


@traced
def install_ssh_keys_su(user):
    if not exists('~%s/.ssh' % (user,)):
        sudo('mkdir ~%s/.ssh' % (user,))
    content = open(SSH_KEY).read().strip()
    local = SSH_KEY + '.local'
    if os.path.exists(local):
        content = "\n".join([content, open(local).read().strip(), ''])
    put(
        StringIO(content),
        '/home/%s/.ssh/authorized_keys' % (user,),
        mode=0o644,
        use_sudo=True,
    )
    sudo('chown -R %(user)s:%(user)s ~%(user)s/.ssh' % locals())
    sudo('chmod 0700 ~%(user)s/.ssh' % locals())
    env.target_os.selinux_restore_context('/home')


@traced
def firmware_generic_symlinks(product=None, reverse=None):
    if reverse is None:
        reverse = REVERSE_LINKS
    product = product or env.product
    for location in ['opt', 'etc', 'var']:
        target = os.path.join('/', location, product)
        link = os.path.join('/', location, 'firmware')
        if reverse:
            log.info("Setting up reverse linking for product")
            target, link = link, target
        if exists(link) and exists(target):
            if not islink(link) and not islink(target):
                log.warning(
                    "Both link and target are directories, moving everything to %s",
                    target,
                )
                sudo('rsync -av %(link)s/* %(target)s/' % locals())
                sudo('rm -rf %(link)s' % locals())
            elif islink(target):
                log.warning("Target is the link currently moving to %s", target)
                sudo('rm %(target)s' % locals())
                sudo('mv %(link)s %(target)s' % locals())
        sudo('mkdir -p %(target)s' % locals())
        base = os.path.basename(target)
        sudo('ln -T -f -s ./%(base)s %(link)s' % locals())
        if location == 'var':
            sudo('chown -R %s %s' % (env.user, target))
            sudo('chown -R %s %s' % (env.user, link))


MIGRATION_PIPS = [
    'django==1.6.8',
    'south==1.0.2',
    'netifaces',
    'requests==2.3.0',
    'django_nose',
    'wheel',
    'fussy==1.0.13',
    'fitting==1.0.5',
    'git+https://github.com/mcfletch/webassets#egg=webassets',
    'git+https://github.com/mcfletch/django-assets#egg=django_assets',
]


@traced
def upgrade_migration_pips():
    upgrade_pips(MIGRATION_PIPS)


@traced
def migration_virtualenv(environment=None):
    """Create a virtualenv inside environment that *just* does south migrations"""
    if env.target_os.name != 'precise':
        # don't build migration virtualenv on non-precise boxes...
        return
    if env.product not in ('digistream', 'epgdata', 'configserver'):
        return
    pips = MIGRATION_PIPS
    if environment is None:
        environment = env.virtual_env
    sub_environment = os.path.join(environment, 'retiresouth')
    with settings(warn_only=True):
        sudo('rm -rf %s' % (sub_environment))
    virtualenv(
        sub_environment,
        [
            os.path.join(SOURCE_ROOT, 'retiresouth'),
        ],
        pips=pips,
    )
    # Strip the code...
    target_python = TARGET_PYTHON
    legacy_mode = '-b' if target_python.startswith('python3') else ''
    sudo(
        '%(target_python)s -m compileall %(legacy_mode)s %(sub_environment)s/lib/%(target_python)s/site-packages'
        % locals()
    )
    sudo(
        'rm -rf $(find %(sub_environment)s/lib/%(target_python)s/site-packages -name "*.py")'
        % locals()
    )


@traced
def empty_virtualenv(
    environment=None,
    need_system_flag=True,
    pip_params='',
    target_python=TARGET_PYTHON,
):
    if environment is None:
        environment = env.virtual_env
    user = env.user
    LOCAL_BIN = (
        'PYTHONPATH=/home/%(user)s/.local/lib/%(target_python)s/site-packages/ '
        + '/home/%(user)s/.local/bin/'
    ) % locals()
    final_target = '/usr/bin/%s' % (target_python,)
    if env.target_os.name == 'precise':
        custom_bin = '/opt/%s/current/bin' % (env.product,)
        custom_python = os.path.join(custom_bin, target_python)
        if exists(custom_python):
            # CUSTOM INSTALL IS IN PATH
            LOCAL_BIN = custom_bin + '/'
            final_target = custom_python
        elif not exists(os.path.join('/usr/bin/', target_python)):
            ensure_python(target_python)
            with settings(warn_only=True):
                sudo(
                    'apt-get remove -y python-pip python-virtualenv python-setuptools python-distribute'
                )
    else:
        if not exists(os.path.join('/usr/bin/', target_python)):
            ensure_python(target_python)
    with cd('~%s' % (env.user,)):

        with settings(warn_only=True):
            current = run(LOCAL_BIN + 'virtualenv --version' % locals())
        try:
            if 'command not found' in current:
                raise ValueError(current)
            if current.startswith('virtualenv '):
                # newer virtualenvs print the package name first...
                current = current.split()[1]
            current = [int(x) for x in current.split('.')]
        except (ValueError, Exception):
            log.exception("Unable to get current virtualenv release")
            current = None
        if not current or current < [1, 11]:
            log.info("Current virtualenv release too old: %s, installing", current)
            put(GET_PIP, '~/get-pip.py')
            if env.target_os.name == 'precise':
                # Legacy: we can't rely on pip actually finding the ancient versions
                # on real pypi any more, so we use our local instance instead...
                raise RuntimeError(
                    """Should not be doing this, custom install should be used which has modern virtualenv"""
                )
                pip_constraint = '"pip < 10"'
                run(
                    '/usr/bin/%(target_python)s ~/get-pip.py --trusted-host 10.1.0.213 -I --user http://10.1.0.213:8080/packages/pip-9.0.3-py2.py3-none-any.whl#md5=d512ceb964f38ba31addb8142bc657cb http://10.1.0.213:8080/packages/wheel-0.31.0-py2.py3-none-any.whl#md5=240d714477a715bcd90e94cb2c44f28c http://10.1.0.213:8080/packages/setuptools-39.2.0-py2.py3-none-any.whl#md5=8d066d2201311ed30be535b473e32fed https://files.pythonhosted.org/packages/85/d8/9dbb4d0fc19f029a43f4b5290b91da87f86572585f5c9b3ed345c6d9b8d7/virtualenv-16.7.12-py2.py3-none-any.whl'
                    % locals()
                )
            else:
                pip_constraint = '"pip >= 10"'
                run(
                    '/usr/bin/%(target_python)s ~/get-pip.py -I --user %(pip_constraint)s setuptools virtualenv'
                    % locals()
                )
        with settings(warn_only=True):
            if not exists(environment):
                sudo(
                    LOCAL_BIN
                    + 'virtualenv --python %s %s'
                    % (
                        final_target,
                        environment,
                    )
                )
                # if env.target_os.name == 'precise':
                #     sudo('ln -s ../env %s/local' % (env.virtual_env,))
                #     sudo('%s/bin/pip install wheel' % (env.virtual_env))


@traced
def wheel_install(*requirements):
    """Build and install wheels for a set of requirements files"""
    if env.target_os.name != 'precise':
        checkout_binary_components()
    with tempdir(prefix='wheel-', suffix='-house') as wheelhouse:
        arguments = []
        directs = False
        # TODO: use osupdates.reqs to load this...
        for requirement in requirements:
            if requirement.startswith('-r'):
                requirement = requirement[2:].strip()
            elif requirement.startswith('-e'):
                requirement = requirement[2:].strip()
            if (
                os.path.isfile(requirement)
                and os.path.splitext(requirement)[1] == '.txt'
            ):  # TODO: could guard against accidents here...
                arguments.extend(['-r', requirement])
            elif os.path.isfile(requirement) and '/wheels/' in requirement:
                log.info("Directly copying binary wheel %s", requirement)
                shutil.copy(requirement, wheelhouse)
                directs = True
            else:
                arguments.append(requirement)
        if arguments:
            build_wheels(wheelhouse, arguments)
        if arguments or directs:
            push_wheels(wheelhouse)
        else:
            log.info("Nothing was selected for installation")


@traced
def build_wheels(wheelhouse, arguments):
    arguments = " ".join(arguments)
    extra_index = 'file://' + os.path.abspath(
        os.path.join(
            SOURCE_ROOT, 'vadagst', 'binary-components', 'wheels', env.target_os.name
        )
    )
    venv = '/tmp/wheel-env-%s' % (TARGET_PYTHON)
    if os.path.exists(venv):
        shutil.rmtree(venv)
    local('VIRTUALENV_NEVER_DOWNLOAD=1 virtualenv -p %s %s' % (TARGET_PYTHON, venv))
    if env.target_os.name == 'precise':
        local('%s/bin/pip install -U "pip<10" "wheel<0.32.0"' % (venv))
    else:
        # Pinned due to regression in 18.1
        local('%s/bin/pip install -U pip wheel' % (venv))
    local(
        '%(venv)s/bin/pip wheel --isolated -f %(extra_index)s -w %(wheelhouse)s %(arguments)s'
        % locals()
    )
    return wheelhouse


@traced
def push_wheels(wheelhouse):
    """Push a wheelhouse (directory of binary wheels) to the target machine"""
    remote = '/tmp/wheels/'
    rsync_project(remote, wheelhouse + '/', delete=True)
    with settings(warn_only=True):
        # ARGH, don't look, this is crazy, but python crashes re-compiling bytecode when the
        # bytecode comes from a newer Python (e.g. 2.7.14 vs 2.7.5)... which is some serious
        # WTF-ery
        log.info("Doing the forced installation")
        sudo(
            '%s/bin/pip install -I --no-deps --no-index %s*.whl'
            % (env.virtual_env, remote)
        )
    # The second call *is* checked, and verifies that all of the wheels really were installed,
    # even if the post-installation re-compilation has failed...
    # Sigh, can't do this anymore, as pip has decided to fix *this*
    # but old platforms may have old pips...
    # log.info("Checking that all wheels were installed by doing an optional install")
    # sudo('%s/bin/pip install %s*.whl' % (env.virtual_env, remote))
    # log.info("Force installation succeeded")


@traced
def find_pods(hostname):
    pod_set = json.loads(
        subprocess.check_output(['kubectl', 'get', 'pods', '-o', 'json'])
    )
    pods = []
    for pod in pod_set['items']:
        if pod['spec'].get('hostname') == hostname:
            pods.append(pod['metadata']['name'])
    return pods


@traced
def azure_push(wheelhouse, podname):
    """Push a wheelhouse directory to azure pods matching podname"""
    remote = '/code/wheels'
    base_match = re.compile('^(?P<base>[a-zA-Z][a-zA-Z0-9-]*)-\d+[.]\d+[.]\d+')
    for pod in find_pods(podname):
        local(
            'kubectl exec -it %(pod)s -- /bin/bash -c "mkdir -p %(remote)s && rm -rf %(remote)s/*.whl"'
            % locals()
        )
        names = []
        for filename in glob.glob(os.path.join(wheelhouse, '*.whl')):
            bm = base_match.match(os.path.basename(filename))
            if bm:
                base = bm.group('base')
                names.append(base)  # strip .whl
            else:
                raise RuntimeError("Could not find base in %r" % (filename,))
            local('kubectl cp %(filename)s %(pod)s:%(remote)s' % locals())
        names = " ".join(names)
        local(
            'kubectl exec -it %(pod)s -- /bin/bash -c "pip uninstall -y %(names)s && pip install --no-deps %(remote)s/*.whl"'
            % locals()
        )


def devpi_prebuilt(use_prebuilt=False):
    if use_prebuilt:
        extra_index = (
            '--trusted-host 10.1.0.213 --extra-index-url=http://10.1.0.213:8080/simple/'
        )
    else:
        extra_index = ''
    return extra_index


@traced
def assemble_sources(
    requirements,
    workdir='/tmp/source-build',
    deps=True,
    filter_packages=None,
    run=run,
    use_prebuilt=False,
):
    """Given a requirements file collect source-code for all projects

    Runs pip download on the set (excluding binaries) and then uploads
    to the remote path workdir.
    """
    from osupdates import reqs

    sudo(
        'chown -R %(user)s %(workdir)s'
        % {
            'user': env.user,
            'workdir': workdir,
        }
    )
    run('mkdir -p %(workdir)s' % locals())
    run('mkdir -p %(workdir)s/local' % locals())
    with tempdir(prefix='source-', suffix='-packages') as local_working:
        if not isinstance(requirements, list):
            requirements = reqs.load_requirements(requirements)
        sources, remote = [], []
        for x in requirements:
            if x.startswith('-e ') and os.path.isdir(x[3:]):
                sources.append(os.path.abspath(x[3:]))
            elif os.path.isfile(x) and x.endswith('.whl'):
                shutil.copy(x, local_working)
            else:
                if 'git+' in x and '#' in x:
                    x = x.split('#')[0]
                if x.startswith('-e '):
                    x = x[3:]
                remote.append(x.strip('"'))

        source_files = []
        with cd(workdir):
            local('rm -rf %(workdir)s/*' % locals())
            for source in sources:
                with lcd(source):
                    local(
                        "python setup.py sdist --dist-dir=%(local_working)s" % locals()
                    )
            for i in range(5):
                try:
                    project.rsync_project(
                        workdir + '/local/',
                        local_working + '/*',
                        delete=True,
                    )
                except Exception as err:
                    pass
                else:
                    break
            source_files = os.listdir(local_working)
            reqfile = os.path.join(workdir, '.requirements.txt')
            put(StringIO('\n'.join(remote)), reqfile)

            function = [
                '%s/bin/pip' % (env.virtual_env),
                'wheel',
                # '--global-option', '--formats=gztar',
                # '--no-binary',':all:',
                # '--exists-action','i',
                '--find-links',
                'file://%s' % (workdir,),
                '-w',
                workdir,
                '--src',
                os.path.join(workdir, 'sources'),
                devpi_prebuilt(use_prebuilt=use_prebuilt),
                '-r',
                reqfile,
            ] + [os.path.join(workdir, 'local', x) for x in source_files]
            run(' '.join(function))

        # deps_fragment = [] if deps else ['--no-deps']
        # function = [
        #     'pip','download',
        #         '--global-option', '--formats=gztar',
        #         '--no-binary',':all:',
        #         '--exists-action','i',
        #         '--dest', sources,
        # ] + deps_fragment + packages
        # local(' '.join(function))
        # with lcd( sources ):
        #     if filter_packages:
        #         local( 'rm -rf %s'%(
        #             ' '.join( filter_packages )
        #         ))
        # project.rsync_project(
        #     workdir+'/',
        #     sources + '/*',
        #     delete=True,
        # )
    return workdir


@traced
def system_module(module, target_python=TARGET_PYTHON):
    """Link the system gobject-introspection module into the virtualenv"""
    # now make the gi library available to the virtualenv...
    if hasattr(env.target_os, 'ensure_locales'):
        env.target_os.ensure_locales()
    sudo('%s/bin/link-os-module %r' % (env.virtual_env, module))
    # target = run(
    #     '/usr/bin/%(target_python)s -c "import gi;print(gi.__file__)"'%locals()
    # ).strip()
    # target = os.path.dirname( target )
    # link = '%(env)s/lib/%(target_python)s/site-packages/gi'%{
    #     'target':target,
    #     'target_python':TARGET_PYTHON,
    #     'env': env.virtual_env,
    # }
    # if not exists(link):
    #     sudo( 'ln -s %(target)s %(link)s'%locals())


@traced
def virtualenv(
    environment=None,
    project_sources=None,
    pips=None,
    need_system_flag=True,
    pip_params='',
    target_python=TARGET_PYTHON,
    remote_assembly=False,
    use_prebuilt=False,
):
    """Create a virtual environment at environment with pips and project_sources installed"""
    if environment is None:
        environment = env.virtual_env
    if project_sources is None:
        project_sources = []
    if pips is None:
        pips = env.pips
    if exists(environment):
        sudo('rm -rf %s' % (environment,))
    empty_virtualenv(
        environment=environment,
        need_system_flag=need_system_flag,
        pip_params=pip_params,
        target_python=target_python,
    )
    packages = '/tmp/packages/'
    sudo('mkdir -p %(packages)s/.home' % locals())
    sudo('mkdir -p %(packages)s/local' % locals())
    # remove the final package-set to avoid duplicates...
    sudo('rm -f %(packages)s/*.whl' % locals())
    sudo('rm -rf %(packages)s/local/*' % locals())
    # packages = '/home/%s/packages'%( env.user, )
    home = os.path.join('/home', env.user)
    prefix = 'HOME=%s' % packages
    if env.target_os.name == 'precise':
        if exists('/opt/%s/current/bin/python2'):
            pip = ''
        else:
            pip = '"pip < 10"'
    else:
        pip = '"pip>=8.0.2"'
    remote_assembly = True
    if remote_assembly:
        packages = assemble_sources(pips, packages, use_prebuilt=use_prebuilt)
        pips = '%(packages)s/*.whl' % locals()
        sudo(
            '%(prefix)s %(environment)s/bin/pip --disable-pip-version-check install %(pip_params)s -I %(pips)s'
            % locals()
        )
    if pips and not remote_assembly:
        sudo(
            '%(prefix)s %(environment)s/bin/pip --disable-pip-version-check install %(pip_params)s -U %(pip)s "wheel<0.32.0"'
            % locals()
        )
        if not isinstance(pips, (list, tuple)):
            raise RuntimeError("Please use list form")
        log.info("Doing wheel installation")
        wheel_install(*(tuple(pips) + tuple(project_sources)))


@traced
def forceinstall(sources, build_webpack=True):
    """Dev: Force install given project source into environment

    Used to do hot patching of a server with the local version of a package,
    this function installs the *source code* onto the server. As a result it
    might result in the leak of source code if a user were to examine
    the hard-disk of the machine.

    You can separate multiple packages with ':' characters...
    """
    if str(build_webpack).lower() in ('n', 'no', 'f', 'false', 'off', '0'):
        build_webpack = False
    elif str(build_webpack).lower() in ('y', 'yes', 't', 'true', 'on', '1'):
        build_webpack = True
    else:
        raise ValueError(
            'build_webpack needs yes/no or 1/0 or false/true, got %s' % (build_webpack)
        )
    virtual_env = env.virtual_env
    prefix = 'HOME=/root'
    if isinstance(sources, (bytes, unicode)):
        sources = sources.split(':')
    sources = [source if '/' in source else find_dist(source) for source in sources]
    if build_webpack:
        for source in sources:
            log.info("Doing webpack build for %s", source)
            webpack_project(os.path.abspath(source))
    log.info("Force installing: %s", ", ".join(sources))
    with tempdir(prefix='wheel-', suffix='-house') as wheelhouse:
        build_wheels(wheelhouse, ['--no-deps'] + sources)
        push_wheels(wheelhouse)


@traced
def azure_forceinstall(sources, hostname='shogun-django'):
    """Forceinstall sources to azure instance via k8s copy and install..."""
    prefix = 'HOME=/root'
    if isinstance(sources, (bytes, unicode)):
        sources = sources.split(':')
    sources = [source if '/' in source else find_dist(source) for source in sources]
    log.info("Force installing: %s", ", ".join(sources))
    with tempdir(prefix='wheel-', suffix='-house') as wheelhouse:
        build_wheels(wheelhouse, ['--no-deps'] + sources)
        azure_push(wheelhouse, hostname)


@traced
def azure_killall(process='uvicorn', hostname='shogun-django'):
    for pod in find_pods(hostname):
        local('kubectl exec -it %(pod)s -- pkill -f %(process)s' % locals())


@traced
def checkout_binary_components():
    bc = os.path.join(
        SOURCE_ROOT,
        'vadagst',
        'binary-components',
    )
    if not os.path.exists(bc):
        with lcd(os.path.dirname(bc)):
            local(
                'git clone -b master ssh://git@eng.atxnetworks.com/jan/binary-components.git'
            )
    else:
        with lcd(bc):
            local('git fetch')
            local('git checkout master')
            local('git pull --ff-only')
    with lcd(SOURCE_ROOT):
        local('git submodule foreach git pull --ff-only')
    return bc


def build_wheel(source):
    """Dev: build a wheel from the given project source and upload to 10.1.0.224 repo..."""
    virtual_env = env.virtual_env
    if not '/' in source:
        source_directory = find_dist(source)
    else:
        source_directory = source
    print("Compiling: %s" % (source_directory))
    filename = upload_project(source_directory)
    run('mkdir -p ~/wheelhouse')
    run('rm -rf ~/wheelhouse/*.whl')
    run(
        '%(virtual_env)s/bin/pip --isolated wheel -w ~/wheelhouse/ %(filename)s'
        % locals()
    )

    bc = checkout_binary_components()
    target = os.path.join(bc, 'wheels', env.target_os.name)
    print("Downloading to: %s" % (target,))
    get(
        '~/wheelhouse/*.whl',
        target,
    )
    run('rm -rf %s' % (filename))


def install_wheel(name):
    wheel = put(name, '~/')
    sudo(
        '%(virtualenv)s/bin/pip  --disable-pip-version-check  install --force -I --no-deps %(wheel)s'
        % {
            'virtualenv': env.virtual_env,
            'wheel': wheel[0],
        }
    )


@traced
def fussy_clean():
    path = os.path.dirname(env.software_directory)
    virtual_env = env.virtual_env
    with settings(warn_only=True):
        sudo('%(virtual_env)s/bin/fussy-clean -t %(path)s' % locals())


@traced
def build_clean():
    user = env.user
    sudo('rm -rf ~%(user)s/*.gpg' % locals())


@traced
def django_preconfigure():
    sudo('mkdir -p /etc/nginx/keys/')
    product = env.product
    config_app = env.config_app
    user = env.user
    group = env.group
    envs = 'TARGET_OS=%s INIT_SYSTEM=%s' % (
        env.target_os.name,
        env.target_os.init_system.name,
    )
    sudo(
        '%(envs)s /opt/%(product)s/current/env/bin/preconfigure-django %(product)s %(config_app)s %(user)s %(group)s'
        % locals()
    )


def remove_bundles(path):
    for path, subdirs, files in os.walk(path):
        if os.path.basename(path) == 'bundles':
            for filename in files:
                os.remove(os.path.join(path, filename))
            return True
    return False


@traced
def webpack(configs, variant=None):
    """Do a production build of webpack bundles locally"""
    variant = variant or env.default_js_variant
    if variant in ('prod', 'dash-prod'):
        variant = 'production'
    else:
        variant = 'development'

    if configs:
        config = configs[0]
        print("Building webpack config in %s" % (config))
        workdir = os.path.dirname(config)
        with lcd(workdir):
            local('NODE_ENV=%(variant)s webpack --config %(config)s' % locals())
            return True
    log.error("No webpack configuration provided")
    return False


@traced
def django_collectstatic():
    """Collect django static assets (use update_static_files if the product doesn't use assets)"""
    django_admin = env.django_admin
    product = env.product
    with settings(warn_only=True):
        sudo('mkdir -p /opt/%(product)s/current/www/static' % locals())
    sudo('%(django_admin)s collectstatic --clear --noinput ' % locals())
    if 'django_assets' in product_settings().INSTALLED_APPS:
        sudo('%(django_admin)s assets build ' % locals())
    else:
        raise RuntimeError(
            "Fab file is using django_collectstatic, but it should be using update_static_files (dash GUI)"
        )
    sudo('chmod -R go+r /opt/%(product)s/current/www/static' % locals())


@contextmanager
@traced
def with_collectstatic():
    """Collect static files *locally* for deployment to target(s)"""
    product = env.product
    with tempdir(prefix='static-', suffix='-collect') as static_dir:
        log.info("Collecting static files in %s", static_dir)
        workenv = os.environ.copy()
        workenv['STATIC_ROOT'] = static_dir
        workenv['DJANGO_SETTINGS_MODULE'] = '%s.settings' % (product)
        log.info("Doing asset bundle build")
        if 'django_assets' in product_settings().INSTALLED_APPS:
            log.warning("django assets in the install applications, doing assets build")
            nbio.Process(
                [
                    'django-admin',
                    'assets',
                    'build',
                    # '--directory', static_dir,
                    '--manifest',
                    os.path.join(static_dir, '.webassets-manifest'),
                ],
                env=workenv,
            )()
        log.info("Collecting static files into %s", static_dir)
        nbio.Process(
            [
                'django-admin',
                'collectstatic',
                '--no-input',
                '--clear',
            ],
            env=workenv,
        )()
        log.info("Assets collected")
        yield static_dir
        log.info("Cleaning up static files")


@traced
def update_static_files(firmware_directory='/opt/firmware/current', skip_pack=False):
    """Locally pack, collect and then push to given firmware directory"""
    if skip_pack not in ('t', True, 'True', 'T', 'y', 'yes', 'Y'):
        webpack_project(os.path.join(SOURCE_ROOT, env.product))
    with with_collectstatic() as static_dir:
        install_su(
            static_dir,
            os.path.join(firmware_directory, 'www/static'),
            delete=True,
        )
        sudo('chmod -R go+r %(firmware_directory)s/www/static' % locals())


@traced
def hotpatch_javascript():
    """Do all the operations necessary to get local javascript on machine for dev"""
    forceinstall('atxstyle')  # runs webpack too
    django_collectstatic()
    django_killall()


@traced
def hotpatch_webpack():
    """Take local dev bundles and push to the remote machine

    Assumes you are currently running webpack-watch on your
    local machine such that your bundles are the current
    code you want to test. Pushes those bundles to remote
    and restarts django to pick them up.

    Note: this only works for webpack (modern) projects,
    *not* for older ip2av3/burnin/epgdata projects that
    still use django-assets.
    """
    raise RuntimeError(
        "hotpatching webpack needs to be updated to clean bundle structure"
    )
    LOCAL = os.path.join(SOURCE_ROOT, 'atxstyle/atxstyle/static/bundles')
    REMOTE = '/opt/firmware/current/www/static/bundles'
    DIRNAME = '/tmp/bundle-upload'
    rsync_project(
        DIRNAME + '/',
        LOCAL + '/*',
        delete=True,
    )
    sudo('rsync -av --delete %(DIRNAME)s/* %(REMOTE)s/' % locals())
    with settings(warn_only=True):
        django_killall()


def _find_admin_json():
    here = os.path.dirname(
        __import__(env.config_app, {}, {}, env.config_app.split('.')).__file__
    )
    content = nbio.Process('find %(here)s -name "admin_user.json"' % locals())()
    return content.strip()


@traced
def admin_user():
    """Find our admin_user.json file, load it"""
    file = _find_admin_json()
    if not file:
        raise RuntimeError("Could not find admin_user.json")
    loaddata_file(file)


@traced
def loaddata_file(local):
    """Load data from the given filename on the remote server

    This assumes that we do *not* want the file to remain on the server
    """
    temp = '~/tmp/'
    run('mkdir -p %(temp)s' % locals())
    filename = os.path.join(temp, as_unicode(os.path.basename(local)))
    put(as_unicode(local), temp)
    try:
        django_loaddata(filename)
    finally:
        sudo(u'rm -rf %(temp)s' % locals())


@traced
def django_loaddata(file):
    django_admin = env.django_admin
    run('%(django_admin)s loaddata %(file)s' % locals())


@traced
def django_syncdb():
    django_admin = env.django_admin
    run('%(django_admin)s syncdb --noinput ' % locals())


@traced
def django_admin(*args):
    django_admin = env.django_admin
    args = ' '.join(args)
    run('%(django_admin)s %(args)s' % locals())


@traced
def django_killall():
    """Kill all web clients to reload *just* the web GUI"""
    killall('gunicorn')


@parallel
def killall(process):
    run('pkill -f %s' % (process,))


@traced
def lockdown():
    common_etc()
    lock_bootloader()


def _is_not_empty(source_directory):
    if not os.path.exists(source_directory):
        return False
    if not glob.glob(os.path.join(source_directory, '*')):
        return False
    return True


def _template_directories(directory):
    """Decide which template directories match pattern "directory"

    common-config directory
    common-config os-name directory
    for each installed local project in env.REQUIREMENTS:
        project directory
        project os-name directory
    """
    sources = [
        os.path.join(COMMON_CONFIG, directory),
        os.path.join(COMMON_CONFIG, env.target_os.name, directory),
    ]
    if hasattr(env, 'HERE'):
        sources.extend(
            [
                os.path.join(env.HERE, 'opt', env.product, 'current', directory),
                os.path.join(
                    env.HERE,
                    'opt',
                    env.product,
                    'current',
                    env.target_os.name,
                    directory,
                ),
            ]
        )
    if hasattr(env, 'REQUIREMENTS'):
        for pip in load_requirements(env.REQUIREMENTS):
            if pip.startswith('-e '):
                pip = pip[3:].strip()
            # TODO: this should be releative to the source requirements file...
            pip = os.path.abspath(pip)
            if os.path.isdir(pip):
                sources.extend(
                    [
                        os.path.join(pip, directory),
                        os.path.join(pip, env.target_os.name, directory),
                    ]
                )
    return sources


@traced
def _fix_ssh_permissions():
    with settings(warn_only=True):
        sudo('chown -R %(user)s:%(user)s /home/%(user)s/.ssh' % env)
        sudo('chmod -R go-rwx /home/%(user)s/.ssh' % env)
        sudo(
            'chown -R %(user)s:%(user)s %(software_directory)s/home/%(user)s/.ssh' % env
        )
        sudo('chmod -R go-rwx %(software_directory)s/home/%(user)s/.ssh' % env)


@traced
def install_template_dir(directory, target, remote=True):
    """For given directory, compile all local template directories and then install them

    Note: this installs into the /opt/firmware/current/%(target)s directory,
    which will *later* be copied by post-install to the final location.

    Uses _template_directories(directory) to decide which directories to include
    in the final template.
    """
    if remote:
        _fix_ssh_permissions()
    short_name = os.path.basename(directory)
    with tempdir(prefix='template-', suffix='-%(short_name)s' % locals()) as temp:
        log.info(
            "Compiling %s template files",
            directory,
        )
        base = os.path.basename(target)
        build = os.path.join(temp, os.path.basename(target.lstrip('/')))
        os.makedirs(build)
        for possible_source in _template_directories(directory):
            if os.path.isdir(possible_source):
                local_target = build
                if os.path.exists(os.path.join(possible_source, 'product-user.txt')):
                    local_target = os.path.join(build, env.user)
                    if not os.path.exists(local_target):
                        os.makedirs(local_target)
                log.info("Adding %s => %s", possible_source, local_target)
                for filename in os.listdir(possible_source):
                    src = os.path.join(possible_source, filename)
                    dst = os.path.join(local_target, filename)
                    if os.path.isdir(src):
                        # Note: filename is often multiple segments...
                        # log.info("  D %s => %s", src, dst)
                        parent_dir = os.path.dirname(dst)
                        if not os.path.exists(parent_dir):
                            os.makedirs(parent_dir)
                        nbio.Process(['rsync', '-av', src, parent_dir])()
                    else:
                        # log.info("  F %s => %s", src, dst)
                        if os.path.exists(dst):
                            os.remove(dst)
                        shutil.copyfile(src, dst)
        if os.listdir(build):
            # import ipdb;ipdb.set_trace()
            if remote:
                install_su(
                    build,
                    os.path.join(env.software_directory, target),
                )
                if directory == 'home':
                    _fix_ssh_permissions()
            else:
                log.info("Local: %s => %s", build, target)
                nbio.Process(
                    ' '.join(['rsync', '-av', os.path.join(build, '*'), target + '/'])
                )()


def default_common_directories(firmware_prefix):
    directories = [
        ('etc', os.path.join(firmware_prefix, 'etc')),
        ('sbin', os.path.join(firmware_prefix, 'sbin')),
        ('home', os.path.join(firmware_prefix, 'home')),
    ]
    return directories


@traced
def compile_etc(root='docker-dev/current'):
    result = common_etc(root, firmware_embed_only=False, remote=False)
    local_update_supervisor_username(os.path.join(root, 'etc/supervisor/conf.d'))
    return result


@traced
def common_etc(root='', firmware_embed_only=True, directories=None, remote=True):
    """Install common /etc/ configs"""
    if firmware_embed_only:
        firmware_prefix = 'opt/firmware/current'
    else:
        firmware_prefix = ''
    write_release()
    directories = directories or default_common_directories(firmware_prefix)
    for directory, target in directories:
        log.info("Put: %s => %s/%s", directory, root, target)
        install_template_dir(
            directory,
            os.path.join(root or '/', target.lstrip('/')),
            remote=remote,
        )
    product = env.product
    if remote:
        with settings(warn_only=True):
            # Only present if we have already run post-install
            sudo('chmod -R 0440 %(root)s/etc/sudoers.d/promotions' % locals())
            sudo('chown -R root:root %(root)s/opt' % locals())
            sudo('chown -R root:root %(root)s/etc/%(product)s' % locals())
            sudo('chown -R root:root %(root)s/etc/atxlicense' % locals())
            sudo('chown -R root:root %(root)s/etc/firmware' % locals())
            sudo('chown -R root:root %(root)s/etc/snmp' % locals())
            # Fixing/confirming permissions on the target for crtical bits...
            sudo('chmod 0755 %(root)s/%(firmware_prefix)s/sbin/*' % locals())
            sudo(
                'chmod 0440 %(root)s/%(firmware_prefix)s/current/etc/sudoers.d/*'
                % locals()
            )


@traced
def update_grub():
    sudo('update-grub2')


@traced
def lock_bootloader(
    username='root',
    password_hash='''5CF08D57C400443B011AC99E568EBFCC04BECA0AF7094D239C9881794A42F455245113FC3E6AE3E28BD7B426CA68BD4E732C78D12004FC8AB54381AAB013DE5E.BDEAC4DADCDE28DEE35775D1DD6CA7A37DED858E3D15B1BBC5CF2CC2A984117AC09B0579B8001576D4C65B9AA92D7254C4617E50FBF1CE2A57CB1BC85F918012''',
):
    contents = (
        '''
cat << EOF
set superusers="%(username)s"
export superusers
password_pbkdf2 %(username)s grub.pbkdf2.sha512.10000.%(password_hash)s
EOF
'''
        % locals()
    )
    user_config = '/etc/grub.d/01_user_list'
    env.target_os.write_string_to_file(contents, user_config)
    sudo('chmod +x %(user_config)s' % locals())
    env.target_os.update_grub()


@traced
def disable_ipv6():
    ensure_lines_in_file(
        '/etc/sysctl.conf',
        [
            'net.ipv6.conf.lo.disable_ipv6 = 1',
            'net.ipv6.conf.all.disable_ipv6 = 1',
            'net.ipv6.conf.default.disable_ipv6 = 1',
        ],
    )


def pattern_in_lines(pattern, lines):
    if isinstance(pattern, (bytes, unicode)):
        return pattern in lines and pattern
    else:
        for line in lines:
            if pattern.search(line):
                return line


def ensure_lines_in_file(filename, required=None, disallowed=None):
    lines = sudo('cat %(filename)s' % locals()).splitlines()
    changed = False
    for line in disallowed or ():
        matched = pattern_in_lines(line, lines)
        while matched:
            lines.remove(matched)
            changed = True
            matched = pattern_in_lines(line, lines)
    for line in required or ():
        if not pattern_in_lines(line, lines):
            lines.append(line)
            changed = True
    if changed:
        env.target_os.write_string_to_file('\n'.join(lines), filename)


@traced
def install_su(source_dir, target_dir, delete=False, owner='root', **named):
    """Install source directory into target directory on server

    Uses rsync_project internally, the **named parameters are
    passed through, with extra_opts populated to match the "delete" parameter
    """
    temp = '~%s/tmp/' % (env.user,)
    run('mkdir -p %(temp)s' % locals())
    print('Install su: %s -> %s' % (source_dir, target_dir))
    with cd(temp):
        base = os.path.basename(source_dir)
        user = env.user

        with settings(warn_only=True):
            sudo('rm -rf %(temp)s%(base)s' % locals())
        if delete:
            named['extra_opts'] = named.get('extra_opts', '') + ' -l --delete-after'
            rsync_project('%(temp)s' % locals(), source_dir, **named)
        else:
            named['extra_opts'] = named.get('extra_opts', '') + ' -l'
            rsync_project('%(temp)s' % locals(), source_dir, **named)
        try:
            sudo('chown -R %(owner)s:%(owner)s %(temp)s%(base)s' % locals())
            sudo('mkdir -p %s' % (target_dir))
            sudo('rsync -alv %(temp)s%(base)s/* %(target_dir)s' % locals())
            dots = [x for x in os.listdir(source_dir) if x.startswith('.')]
            if dots:
                sources = ' '.join(
                    [
                        '%(temp)s%(base)s/%(name)s'
                        % {
                            'temp': temp,
                            'base': base,
                            'name': name,
                        }
                        for name in dots
                    ]
                )
                sudo('rsync -alv %(sources)s %(target_dir)s' % locals())
        finally:
            # now switch the template back for next time...
            with settings(warn_only=True):
                sudo('chown -R %(user)s %(temp)s%(base)s' % locals())


def path_to_user(path, user, group=None):
    """Convert path to being owned by user (and group)"""
    owner = user
    if group:
        owner = '%s:%s' % (user, group)
    sudo('chown -R %(owner)s %(path)s' % locals())


def user_to_group(user, group):
    env.target_os.user_to_group(user, group)


@parallel
def reboot():
    run('sudo /sbin/reboot')


def print_revno():
    print(revno())


def release_metadata(settings=env):
    return {
        'rev': os.environ.get('BAMBOO_VERSION') or revno(),
        'hash': commit_hash(),
        'branch': commit_branch(),
        'product': settings.product,
        'user': settings.user,
        'os': settings.target_os.name,
    }


def release_file_content():
    metadata = release_metadata()
    return (
        '''[release]
revision=%(rev)s
commit=%(hash)s
branch=%(branch)s
product=%(product)s
os=%(os)s
'''
        % metadata
    )


def local_release_file():
    content = release_file_content()
    from fussy import twrite

    twrite.twrite(
        'docker-dev/release.conf',
        content,
    )


@traced
def write_release():
    dirname = env.product if not REVERSE_LINKS else 'firmware'
    filename = os.path.join('opt', dirname, 'current', 'etc', dirname, 'release.conf')
    with open(filename, 'w') as fh:
        fh.write(release_file_content())
    return filename


def resolve_upstream(upstream):
    """Resolve (package,source) tuples into final package directories

    If the package is not available, will do a pip -e installation first
    """
    result = []
    for item, source in upstream:
        try:
            result.append(find_dist(item))
        except RuntimeError as err:
            err.args += (item,)
            raise
        except ImportError as err:
            expected = os.path.normpath(
                os.path.join(os.path.dirname(__file__), '..', '..', '..', item)
            )
            if not os.path.exists(expected):
                nbio.Process(source, cwd=os.path.dirname(expected))()
            result.append(find_setup(expected))
    return result


def find_dist(name):
    """Find the full path of the latest sdist of name"""
    try:
        package_dir = os.path.dirname(__import__(name).__file__)
    except ImportError as err:
        err.args += (name,)
        raise
    if 'retiresouth' in package_dir and name != 'retiresouth':
        raise ValueError(
            "%s currently links to %s, re-install locally" % (name, package_dir)
        )
    return find_setup(package_dir)


def find_setup(package_dir):
    """Find setup.py file in the parents of package_dir"""
    name = package_dir
    while package_dir and package_dir != '/':
        packer = os.path.join(package_dir, 'setup.py')
        if os.path.basename(package_dir) == 'site-packages':
            raise RuntimeError(
                "%s appears to be an installed package, not a source package" % (name,)
            )
        elif os.path.exists(packer):
            return package_dir
        package_dir = os.path.dirname(package_dir)
    raise RuntimeError("Unable to find package for %s" % (name,))


@traced
def webpack_project(project_source, variant=None):
    variant = variant or env.default_js_variant
    available = os.path.join(os.path.abspath(project_source), 'webpack.config.js')
    configs = glob.glob(available)
    if configs:
        return webpack(configs, variant)
    return False


def upload_project(project_source, build_webpack=True):
    """Find the setup.py, build the package, copy and install"""
    build_webpack = distutils.util.strtobool(str(build_webpack))
    if build_webpack:
        webpack_project(project_source)
    try:
        nbio.Process('cd %(project_source)s && python setup.py sdist' % locals())()
    except Exception as err:
        err.args += (project_source,)
        raise
    files = glob.glob(os.path.join(project_source, 'dist', '*.tar.gz'))
    files.sort(key=lambda f: os.stat(f).st_ctime)
    current = files[-1]
    base = os.path.basename(current)
    run('mkdir -p ~/tmp')
    file = os.path.join('~/tmp', base)
    return put(current, file)[0]


def abslink(install_dir, target_dir='/'):
    """Create abs symlink

    For every relative *file* in source_dir (from source_dir)
    Create a symlink from target_dir -> install_dir
    Creating any missing directories
    """
    product = env.product
    sudo(
        '/opt/%(product)s/current/env/bin/fussy-link-tree %(install_dir)s %(target_dir)s'
        % locals()
    )


def ps(arg="auxf"):
    run('ps %(arg)s' % locals())


def patch(file, patchfile, context=True):
    patchfile = os.path.join(PATCHES, patchfile)
    user = env.user
    put(patchfile, '/home/%(user)s/patch.diff' % locals())
    if context:
        context = '-c'
    else:
        context = ''
    sudo('patch --forward %(context)s -i /home/%(user)s/patch.diff %(file)s' % locals())


@traced
def pack_firmware(sku='edgeqam', product=None, version=None, metadata_files=None):
    """Pack currently-built release as firmware (automatically named)

    Deletes source-code (strip_firmware)
    Runs fussy-pack with *no* signing...
    Uses BAMBOO_VERSION to construct a final filename by default (version)
    Downloads the resulting build to the builds directory
    Signs it locally
    """
    if product is None:
        product = env.product
    version = version or os.environ.get('BAMBOO_VERSION')
    user = env.user
    group = env.group
    current = '/opt/%(product)s/current' % locals()
    sudo('chown -R root:root %(current)s' % locals())
    with settings(warn_only=True):
        link_target = run('readlink %(current)s' % locals())
        if not link_target.failed:
            if link_target != current:
                sudo('rm %(current)s' % locals())
                sudo('mv %(link_target)s %(current)s' % locals())
    name = ts_name(sku)
    full_path = os.path.join('/opt/%(product)s' % locals(), name)
    with settings(warn_only=True):
        sudo('rm -rf %(full_path)s' % locals())
    # create a link from current to the relative full path...
    # so that even if we are in a chroot we get the right link...
    sudo(
        'mv /opt/%(product)s/current %(full_path)s && ln -s %(name)s /opt/%(product)s/current'
        % locals()
    )

    strip_firmware(full_path)

    with settings(warn_only=True):
        sudo('rm -rf /opt/%(product)s/%(name)s/current' % locals())

    tar_name = '%s.tar.gz' % (name,)

    output = sudo(
        '/opt/%(product)s/current/env/bin/fussy-pack --unsigned --tarname=%(tar_name)s -r /opt/%(product)s/%(name)s'
        % locals()
    )
    filename = output.strip().splitlines()[-1]
    sudo('mv %(filename)s /home/%(user)s/' % locals())
    dirname = os.path.dirname(filename)
    sudo('rm -rf %(dirname)s' % locals())

    target_filename = os.path.join(
        '/home/%(user)s' % locals(), os.path.basename(filename)
    )
    sudo('chown %(user)s:%(group)s %(target_filename)s' % locals())
    if not os.path.exists('builds'):
        os.makedirs('builds')
    get(target_filename, 'builds')
    sudo('rm %(target_filename)s' % locals())
    basename = os.path.basename(target_filename)
    local_filename = os.path.join('builds', basename)
    with settings(warn_only=True):
        local('fussy-sign -k E71D21F6 -f %(local_filename)s' % locals())
    release = []
    for fn in [
        '/opt/firmware/current/etc/%(product)s/release.conf' % env,
        '/opt/firmware/current/etc/firmware/release.conf' % env,
        '/opt/firmware/current/arcos/Arcos3ScreensApp/release.conf',
    ] + (metadata_files or []):
        with settings(warn_only=True):
            if exists(fn):
                release.append(run('cat %s' % fn).strip())
    if not release:
        raise RuntimeError("Unable to find any release files")
    import datetime

    timestamp = datetime.datetime.utcnow().isoformat()
    release.append(
        '''[package]
filename=%(basename)s.gpg
timestamp=%(timestamp)sUTC
ci_version=%(version)s
'''
        % locals()
    )
    with open(local_filename + '.gpg.release.txt', 'w') as fh:
        fh.write('\n'.join(release))
    return name


@traced
def strip_firmware(full_path, killfile='killfile.globs'):
    """Strip out non-redistributable files from full_path"""
    with settings(warn_only=True):
        for path in open(killfile):
            path = path.strip()
            if path.startswith('#'):
                continue
            else:
                sudo('rm -rf %(full_path)s/%(path)s' % locals())
        for localedir in sudo(
            'find %(full_path)s -name "locale"' % locals()
        ).splitlines():
            unsupported_locales = " ".join(
                sudo(
                    'ls -I "en*" -I "*.py?" -I "*.py" %(localedir)s' % locals()
                ).split()
            )
            if unsupported_locales:
                sudo('cd %(localedir)s && rm -rf %(unsupported_locales)s' % locals())
        # compile and strip...
        target_python = TARGET_PYTHON

        legacy_mode = '-b' if target_python.startswith('python3') else ''
        sudo(
            '%(full_path)s/env/bin/%(target_python)s -m compileall %(legacy_mode)s -q %(full_path)s/env/lib/%(target_python)s/site-packages'
            % locals()
        )
        # work around pysnmp's mib-cache problems
        sudo(
            'rm -rf junk-does-not-exist $(find %(full_path)s/env/lib/%(target_python)s/site-packages -name "*.py" | grep -v migrations | grep -vi mibs)'
            % locals()
        )
        sudo(
            'rm -rf junk-does-not-exist $(find %(full_path)s/env/lib/%(target_python)s/site-packages -name "*.pyc" | grep -ie "migrations\|mibs")'
            % locals()
        )
        # clean out pycache directories...
        sudo(
            'rm -rf junk-does-not-exist $(find %(full_path)s/env/lib/%(target_python)s/site-packages -name "__pycache__")'
            % locals()
        )


def latest_build(sku=None):
    return fabmetadata.latest_build(sku=sku or env.product)


@traced
def testrelease(filename=None, sku=DEFAULT_SKU, manual=False):
    """Install a local firmware image to a remote system. Arguments: filename=None, sku=DEFAULT_SKU, manual=False"""
    if not filename:
        filename = latest_build(sku=sku)
    base_filename = os.path.basename(filename)
    product = env.product
    user = env.user
    group = env.group
    with settings(warn_only=True):
        sudo('mkdir -p /var/%(product)s/firmware' % locals())
        sudo('mkdir -p /opt/%(product)s' % locals())
        sudo('chown -R %(user)s:%(group)s /var/%(product)s' % locals())
    if filename.startswith('http://') or filename.startswith('https://'):
        pull_firmware(filename, target_os=env.target_os.name)
    else:
        put(
            filename, '/var/%(product)s/firmware/%(base_filename)s' % locals()
        )  # rsync is much slower when the files are new builds...
        run(
            'cp /var/%(product)s/firmware/%(base_filename)s /var/%(product)s/firmware/new'
            % locals()
        )
    if manual:
        manual_install(base_filename, sku=sku)
    else:
        promote_install()


FIRMWARE_SUFFIX = '.tar.gz.gpg'


@traced
def manual_install(base_filename=None, sku=DEFAULT_SKU):
    if not base_filename:
        filename = latest_build(sku=sku)
        base_filename = os.path.basename(filename)
    product = env.product
    with cd('/tmp'):
        dirname = base_filename[: -len(FIRMWARE_SUFFIX)]
        if exists(dirname):
            sudo('rm -rf %(dirname)s' % locals())
        with settings(warn_only=True):
            sudo('gpg -d /var/%(product)s/firmware/new | tar -zx' % locals())
        target = os.path.join('/opt', product, dirname)
        if exists(target):
            sudo('rm -rf %(target)s' % locals())
        sudo('mv %(dirname)s %(target)s' % locals())
        with cd('/opt/%(product)s' % locals()):
            sudo('rm -rf current')
            sudo('ln -f -s %(dirname)s current' % locals())
    post_install(build=True)


@parallel
@traced
def promote_install():
    run('sudo /opt/firmware/current/sbin/promote-install')


@traced
def os_updates(build=bool(os.environ.get('BUILD'))):
    product = env.product
    prefix = (
        'BUILD=True TARGET_OS=%s INIT_SYSTEM=%s DJANGO_SETTINGS_MODULE=%s.settings'
        % (
            env.target_os.name,
            env.target_os.init_system.name,
            product,
        )
        if build
        else ''
    )
    sudo(
        'cp /opt/%(product)s/current/etc/%(product)s/release.conf /etc/%(product)s/ || /bin/true'
        % locals()
    )
    sudo('%(prefix)s /opt/%(product)s/current/env/bin/os-updates --force' % locals())


@traced
def post_install(build=bool(os.environ.get('BUILD'))):
    product = env.product
    prefix = (
        'BUILD=True TARGET_OS=%s INIT_SYSTEM=%s'
        % (
            env.target_os.name,
            env.target_os.init_system.name,
        )
        if build
        else ''
    )
    sudo(
        'cp /opt/%(product)s/current/etc/%(product)s/release.conf /etc/%(product)s/ || /bin/true'
        % locals()
    )
    sudo('%(prefix)s /opt/%(product)s/current/.post-install' % locals())
    if not build:
        env.target_os.update_grub()


@traced
def rename_and_reset_network_interfaces():
    """If necessary, rename and reset network interface configuration data"""
    env.target_os.rename_network_interfaces()
    env.target_os.update_grub(force=True)


@traced
def disable_networkmanager():
    env.target_os.disable_networkmanager()


@traced
def clean():
    """Clean the ~/tmp and then run build_clean()"""
    sudo('rm -rf ~/tmp')
    build_clean()


def red5_webapps():
    product = env.product
    install_su(RED5_APPS, '/opt/%(product)s/current/usr/' % locals())


def upgrade_pips(pips=None):
    """Pull source code for packages into our package cache"""
    print("Skipping upgrade pips (uses wheel now)")
    return
    if pips is None:
        pips = env.pips
    environment = env.virtual_env
    if not exists(environment):
        empty_virtualenv(environment)

    pip_params = ''  # -M --mirrors=http://b.pypi.python.org'
    user = env.user
    home = os.path.join('/home', env.user)
    packages = os.path.join(home, 'packages')
    if pips:
        pips = [x for x in pips if not x.startswith('git+')]
        if isinstance(pips, (list, tuple)):
            pips = ' '.join(pips)
    if env.target_os.name == 'precise':
        pip = '"pip < 10"'
    else:
        pip = '"pip>=8.0.2"'
    sudo(
        '%(environment)s/bin/pip install -U %(pip)s "wheel<0.32.0" "setuptools"'
        % locals()
    )
    sudo('%(environment)s/bin/pip wheel -w %(packages)s %(pips)s' % locals())


def versative_drivers(version='10.6.1a12'):
    run('mkdir -p ~/tmp')
    rsync_project(
        '~/tmp/desktopvideo-%(version)s-amd64.deb' % locals(),
        os.path.join(SIDELOADS, 'desktopvideo-%(version)s-amd64.deb' % locals()),
    )
    user = env.user
    sudo('dpkg -i /home/%(user)s/tmp/desktopvideo-%(version)s-amd64.deb' % locals())


def create_template_postgis():
    put(INSTALL_POSTGIS, '~/')
    sudo(
        '/bin/bash /home/%s/%s' % (env.user, os.path.basename(INSTALL_POSTGIS)),
        user='postgres',
    )


@traced
def create_postgres_db(databases=None):
    """Given settings.DATABASES create db user and database"""
    if databases is None:
        databases = product_settings().DATABASES
    env.target_os.create_postgres_databases(databases)
    # env.target_os.sudo( 'mkdir -p /var/%s'%env.product,)
    # env.target_os.sudo( 'chown -R %s:%s /var/%s'%(env.user,env.group,env.product,))
    # with settings(warn_only=True):
    #     if not exists('/var/%s/last_south_migration'%(env.product,)):
    #         env.target_os.write_string_to_file( '1970-01-01 00:00', '/var/%s/last_south_migration'%(env.product,) )


@traced
def create_postgres_db_local(databases=None):
    """Given settings.DATABASES create db user and database locally"""
    if databases is None:
        databases = product_settings().DATABASES
    ostarget.local_target().create_postgres_databases(databases)
    # for name,database in databases.items():
    #     if not ('postgis' in database['ENGINE'] or 'psycopg2' in database['ENGINE']):
    #         raise RuntimeError( 'Can only set up postgres/postgis dbs currently' )
    #     if 'postgis' in database['ENGINE']:
    #         #local( 'sudo -u postgres "/bin/bash %s"'%INSTALL_POSTGIS )
    #         database['TEMPLATE'] = '-T template_postgis'
    #     else:
    #         database['TEMPLATE'] = ''
    #     with settings( warn_only = True ):
    #         users = nbio.Process(
    #             'echo "select usename from pg_user" | psql -t -A template1'
    #         )().splitlines()
    #         if database['USER'] not in users:
    #             local(
    #                 '''sudo -u postgres psql -c "create user %(USER)s CREATEDB encrypted password \'%(PASSWORD)s\'"'''%database,
    #             )
    #         if database['NAME'] not in nbio.Process('echo "select datname from pg_database" | psql -t -A template1')().splitlines():
    #             local( 'sudo -u postgres createdb %(TEMPLATE)s  -E "UTF-8" -O %(USER)s %(NAME)s'%database )


@traced
def install_timescaledb_apt():
    """Install timescale db"""
    sudo(
        'apt-get install -y gnupg postgresql-common apt-transport-https lsb-release wget'
    )
    sudo('/usr/share/postgresql-common/pgdg/apt.postgresql.org.sh')
    sudo(
        'echo "deb https://packagecloud.io/timescale/timescaledb/ubuntu/ $(lsb_release -c -s) main" > /etc/apt/sources.list.d/timescaledb.list'
    )
    sudo(
        'wget --quiet -O - https://packagecloud.io/timescale/timescaledb/gpgkey | apt-key add -'
    )
    sudo('apt-get update --yes')
    sudo('apt-get install --yes timescaledb-2-postgresql-13')


@traced
def add_timescale_db(name='stats', memory='1GB', version=None):
    # setup the server/service for timescaledb...
    # assumes centos7 for now...
    with cd('/tmp'):
        db = product_settings().DATABASES[name]
        env.target_os.add_timescale_db(db, memory=memory, version=version)


@traced
def postgres_backup_databases(databases=None):
    if databases is None:
        databases = product_settings().DATABASES
    ts = time.strftime('%Y-%m-%d-%H-%M-%S', time.gmtime())
    if not os.path.exists('prod'):
        os.mkdir('prod')
    dbs = []
    for name, database in databases.items():
        backup_name = '%s-%s-%s.sql.gz' % (name, env.host, ts)
        if not ('postgis' in database['ENGINE'] or 'psycopg2' in database['ENGINE']):
            raise RuntimeError('Can only set up postgres/postgis dbs currently')
        with settings(warn_only=True):
            sudo(
                'pg_dump %s | gzip -9 -c > /tmp/%s' % (database['NAME'], backup_name),
                user='postgres',
            )
            sudo(
                'mv /tmp/%s /home/%s/'
                % (
                    backup_name,
                    env.user,
                )
            )
            get(
                '/home/%s/%s'
                % (
                    env.user,
                    backup_name,
                ),
                'prod',
            )
            dbs.append(backup_name)
            sudo('rm /home/%s/%s' % (env.user, backup_name))
    return dbs


@traced
def postgres_restore_database(source, databases=None):
    """Given backup source, put that database on the host"""
    if databases is None:
        databases = product_settings().DATABASES
    name = os.path.basename(source).split('-')[0]
    assert name in databases
    db_name = databases[name]['NAME']
    put(source, '/home/%s/restore.sql.gz' % (env.user,))
    with settings(warn_only=True):
        sudo(
            'dropdb %s' % (db_name,),
            user='postgres',
        )
    create_postgres_db({name: databases[name]})
    sudo(
        'gunzip -c /home/%s/restore.sql.gz | psql %s' % (env.user, db_name),
        user='postgres',
    )


def product_settings():
    full_name = '%s.settings' % (env.config_app,)
    return __import__(full_name, {}, {}, full_name.split('.'))


def pull_db():
    for database in postgres_backup_databases(product_settings().DATABASES):
        yield database


def db_by_db_name(base_name):
    for name, settings in product_settings().DATABASES.items():
        if name == base_name:
            return settings
        elif settings.get('NAME') == base_name:
            return settings
    return None


def install_db(backup, pre_restore=None, with_timescale=False):
    backup = os.path.expanduser(backup)
    backup = os.path.normpath(backup)
    if not os.path.exists(backup):
        possible = os.path.join('prod', backup)
        if os.path.exists(possible):
            backup = possible
        else:
            raise RuntimeError("Could not find %r" % (backup,))
    base_name = os.path.basename(backup).split('-')[0]
    database = db_by_db_name(base_name)
    db_name = database['NAME']
    db_owner = database['USER']
    with settings(warn_only=True):
        local('dropdb %(db_name)s' % locals())
    local('createdb -O %(db_owner)s %(db_name)s' % locals())
    if pre_restore:
        pre_restore()
    if backup.endswith('.gz'):
        # the filter here is to allow for restoring postgres 10+ onto
        # machines running < 10
        local(
            "gunzip -c %(backup)s | sed -e '/AS integer/d'  | psql %(db_name)s"
            % locals()
        )
    elif backup.endswith('.pgdump'):
        print('PG Dump restore: %s' % (backup,))
        local(
            'pg_restore -Fc --single-transaction -d %(db_name)s %(backup)s' % locals()
        )
    else:
        local('psql %(db_name)s < %(backup)s' % locals())


def diagnostic_db(diagnostic_sql):
    """Load DB to local machine from a locally-unpacked diagnostics file"""
    install_db(diagnostic_sql, pre_restore=_auth_user_manual)


def _auth_user_manual(SQL=os.path.join(HERE, '..', 'fab-data', 'auth_repair.sql')):
    """Manually recreate user table for a diagnostics dump"""
    database = product_settings().DATABASES['default']
    db_name = database['NAME']
    local('psql %(db_name)s < %(SQL)s' % locals())
    local(
        'echo "ALTER TABLE auth_user OWNER TO %(db_name)s" | psql %(db_name)s'
        % locals()
    )


def replicate_var():
    with lcd('/var/%s' % (env.product,)):
        with settings(warn_only=True):
            local('rm protected/display/status/*.json')
        for directory in ('run', 'media', 'protected', 'arcos'):
            if not exists('/var/firmware/%s' % (directory)):
                continue
            local(
                'rsync -e "ssh -p %s" --exclude=*.rbf --exclude=WebDavLog.txt --exclude=event-log.json.* --exclude=ringbuffer --exclude=ringbuffer-tmp --exclude=tarlogs* --exclude=core --exclude=*.gz --exclude=*.bz2 --exclude=trap.status --exclude=firmware -av %s@%s:/var/firmware/%s ./'
                % (env.port, env.user, env.host, directory)
            )


def replicate_ringbuffer():
    with lcd('/var/%s/media' % (env.product,)):
        for directory in ('ringbuffer',):
            local(
                'rsync -av %s@%s:/var/firmware/media/%s ./'
                % (env.user, env.host, directory)
            )


def replicate_db():
    for db in pull_db():
        install_db(db)


def replicate():
    """Replicate the setup on host locally for testing"""
    replicate_db()
    replicate_var()


def replicate_redis():
    run('redis-cli save')
    dump_path = os.path.join(env.HERE, 'redis-dumps')
    if not os.path.exists(dump_path):
        os.makedirs(dump_path)
    REDIS_DUMP = '/var/lib/redis/dump.rdb'
    get(REDIS_DUMP, dump_path, use_sudo=True)
    local('sudo systemctl stop redis')
    local('sudo mv %s %s' % (os.path.join(dump_path, 'dump.rdb'), REDIS_DUMP))
    local('sudo systemctl start redis')


@traced
def django_create_db():
    django_admin = env.django_admin
    run('%(django_admin)s migrate --traceback' % locals())


@traced
def update_sideloads(testing=False):
    local('sideload-rebuild')
    with lcd(os.path.join(SIDELOADS, '..')):
        if testing:
            target = 'digistream@digi.dev:/var/digistream/media/sideloads/'
        else:
            target = 'digistream@digistreamupdates.atxnetworks.com:/var/epgdata/media/sideloads/'
        local('rsync -av sideloads/* %(target)s' % locals())


def manual_sideloads():
    """Upload sideloads manually to the machine's ~/sideloads/source_files directory"""
    parent = os.path.join('/home', env.user, 'sideloads')
    target = os.path.join(parent, 'source_files') + '/'
    sudo('chown -R %s %s' % (env.user, parent))
    rsync_project(target, SIDELOADS + '/*')
    sudo('chown -R root %s' % (parent,))


def get_sideloads(
    default=os.path.join(
        HERE, '../../osupdates/osupdates/%s-sideloads.txt' % (env.target_os.name,)
    )
):
    """Look for the default sideload set for the product"""
    specific = os.path.abspath(os.path.normpath(getattr(env, 'SIDELOADS', default)))
    if not os.path.exists(specific):
        log.error("Expected a sideload declaration in %r", specific)
        return []
    else:
        sideloads = list(load_dependencies(specific))
        log.info("Loaded %s sideloads from %r", len(sideloads), specific)
        return sideloads


@traced
def expand_sideloads(filenames=None):
    """Get the local sideloads that should be embedded in firmware"""
    if not filenames:
        filenames = []
        for optionset in get_sideloads():
            filenames.extend(optionset)
    elif filenames and isinstance(filenames, (bytes, unicode)):
        filenames = filenames.split(",")

    def sl_paths(filenames):
        return [os.path.join(SIDELOADS, filename) for filename in filenames]

    if filenames:
        paths = sl_paths(filenames)
        for path in paths:
            if path.endswith('.deb.json'):
                set = [
                    record['filename']
                    for record in json.loads(open(path).read())['metadata']
                ]
                paths.extend(sl_paths(set))
            elif path.endswith('.rpm.json'):
                set = [
                    record['file'] for record in json.loads(open(path).read())['rpms']
                ]
                paths.extend(sl_paths(set))
        return paths
    else:
        return []


@traced
def collect_sideloads(target, filenames=None):
    """Collect sideload files into (local) target directory"""
    os.makedirs(target)
    for filename in expand_sideloads(filenames):
        shutil.copy(filename, os.path.join(target, os.path.basename(filename)))


@traced
def embed_sideloads(filenames=None):
    """Embed a collection of sideloads into firmware"""
    with settings(warn_only=True):
        _fix_ssh_permissions()
    paths = expand_sideloads(filenames)

    if paths:
        paths = " ".join(paths)
        user = env.user
        host = env.host
        temp = os.path.join('/home', env.user, 'sideload-staging')
        with settings(warn_only=True):
            run('mkdir -p %(temp)s' % locals())
        extra_args = ''
        if env.port and env.port != 22:
            extra_args = '-e "ssh -p %s"' % (env.port,)
        local(
            'rsync -av %(extra_args)s %(paths)s %(user)s@%(host)s:%(temp)s/' % locals()
        )
        target = os.path.join(
            env.software_directory, 'home', env.user, 'sideloads', 'source_files'
        )

        with settings(warn_only=True):
            sudo('mkdir -p %(target)s' % locals())
        sudo('rsync -av %(temp)s/* %(target)s/' % locals())
        if hasattr(env.target_os, 'verify_sideloads'):
            env.target_os.verify_sideloads(target)
        # don't re-download to local storage, that's silly and wasteful...
        with settings(warn_only=True):
            run('mkdir -p /home/%s/sideloads/source_files/' % (env.user,))
        sudo(
            'rsync -av %s/* /home/%s/sideloads/source_files/ '
            % (
                target,
                env.user,
            )
        )


def force_rpm(filename):
    """Force a particular RPM file to be installed

    Relies on the package *not* being a dependency of anything else
    """
    from osupdates import rpmutils

    working = '/src/binary-components/repo/'
    sudo('mkdir -p %(working)s' % locals())
    sudo('chown %s %s' % (env.user, working))
    with cd(working):
        put(filename, working)
        rpm = rpmutils.parse_rpmname(filename)
        with settings(warn_only=True):
            sudo('yum remove -y %s' % (rpm['package']))
        sudo('yum install -y %s' % (os.path.basename(filename)))


@traced
def fill_in_supervisor_username():
    from fabric.contrib.files import sed

    cmd = (
        "ls -1 "
        "/opt/firmware/current/etc/supervisord.d/*.conf "
        "/opt/firmware/current/etc/supervisor/conf.d/*.conf "
        "2>/dev/null; /bin/true"
    )
    for file in run(cmd).splitlines():
        sed(file, 'digistream[0-9]*', env.user, use_sudo=True, backup='')
    if hasattr(env.target_os, 'move_supervisor'):
        env.target_os.move_supervisor()


DIGISTREAM_MARKER = re.compile('digistream[0-9]*')


@traced
def local_update_supervisor_username(path):
    from fussy import twrite

    for file in glob.glob(os.path.join(path, '*.conf')):
        content = open(file).read()
        new_content = DIGISTREAM_MARKER.sub(env.user, content )

        if content != new_content:
            twrite.twrite(file, new_content)


@traced
def link_mibs():
    """Make ATX MIB (snmp descriptions) files available on standard tool paths"""
    VIRTUAL_ENV = env.virtual_env
    SOFTWARE_DIRECTORY = env.software_directory
    target_python = TARGET_PYTHON
    source_dir = (
        '%(VIRTUAL_ENV)s/lib/%(target_python)s/site-packages/snmpagents/MIBS/'
        % locals()
    )
    if not exists(source_dir):
        log.warning(
            "No snmpagents in the product, not linking mibs (possibly not a front-end product)"
        )
        return
    target = '%(SOFTWARE_DIRECTORY)s/www/static/' % locals()
    with cd(source_dir):
        sudo('tar -zcf PRODUCT-MIBS.tar.gz *.mib')
        sudo('mv PRODUCT-MIBS.tar.gz %(target)s' % locals())
    # Now make the theme MIBs available...
    if exists('%(SOFTWARE_DIRECTORY)s/themes' % locals()):
        result = run('ls %(SOFTWARE_DIRECTORY)s/themes/' % locals())
        if isinstance(result, bytes):
            result = result.decode('utf-8')
        else:
            result = str(result)
        for theme in result.split():
            log.info("Packing theme: %s", theme)
            with cd('%(SOFTWARE_DIRECTORY)s/themes/%(theme)s' % locals()):
                sudo('tar -zcf PRODUCT-MIBS.tar.gz *.mib')


DEVDOC_DIR = os.path.normpath(os.path.join(HERE, '..', '..', 'devdoc'))
DEVDOC_HTML = os.path.join(DEVDOC_DIR, '_build', 'html')


def update_devdocs():
    """Rebuild the devdoc tree

    Note: if this machine has *never* built the tree before, the cross-links
    will not yet work. You have to run the `buildmodule.py` script twice to
    get the cross-links working in all cases.
    """
    source = DEVDOC_HTML
    with lcd(DEVDOC_DIR):
        local('python buildmodule.py')
        local('DJANGO_SETTINGS_MODULE=ip2av3.settings make html')
    upload_devdocs(DEVDOC_HTML)
    return source


def upload_test_fixture(filename):
    """Push a test fixture to the config-server"""
    # TODO: check for existing files first...
    base = os.path.basename(filename)
    local(
        'scp %(filename)s configserver@10.1.0.224:/var/firmware/media/firmware/test-fixtures/'
        % locals()
    )


def upload_devdocs(
    source=DEVDOC_HTML,
    target='configserver@10.1.0.224:/var/firmware/media/firmware/devdocs',
):
    """Upload already-built devdoc tree to configserver@10.1.0.224"""
    local('rsync -av %(source)s/* %(target)s/' % locals())


@traced
def upload_firmware(
    build=None, target='/var/firmware/media/firmware/', sku=DEFAULT_SKU
):
    filename = build or latest_build()
    base = os.path.basename(filename)
    target = os.path.join(target, env.product, base)
    with settings(
        user='configserver',
        host_string='10.1.0.224:22',
        port=22,
    ):
        assert env.user == 'configserver', env
        assert env.host_string == '10.1.0.224:22', env
        run('mkdir -p %s' % (os.path.dirname(target)))
        put(filename, target)
        if os.path.exists(filename + '.release.txt'):
            put(filename + '.release.txt', target + '.release.txt')
    upload_raw_changelog()


@traced
def upload_raw_changelog(target='/var/firmware/media/firmware/rawchangelog.txt'):
    local('git log --format="%cd %h %s" --date=iso > rawchangelog.txt')
    with settings(
        user='configserver',
        host_string='10.1.0.224:22',
    ):
        put('rawchangelog.txt', target)
    local('rm rawchangelog.txt')


@parallel
@traced
def download_firmware(filename=None, source=FIRMWARE_HOST):
    """Lab: Download the given (or current) firmware from the given source

    Instructs the machine to download the given firmware
    (relative path, if None, the last firmware in `./builds/`)
    and immediately install it. This will obviously only work
    within the lab environment.

    See: :py:func:`upload_firmware` to get the build uploaded for
    subsequent download.
    """
    # if env.host_string in source:
    #    print("Skipping %s"%( env.host_string ))
    # else:
    pull_firmware(filename=filename, source=source, target_os=env.target_os.name)
    run('sudo /opt/firmware/current/sbin/promote-install')


def nginx_run_as(username=None, groupname='www-data'):
    from fabric.contrib.files import sed, contains

    filename = '/etc/nginx/nginx.conf'
    if not username:
        stanza = 'user \1 %s;' % groupname
    else:
        stanza = 'user %s %s;' % (username, groupname)
    if contains(filename, 'user '):
        sed(
            filename,
            before='^user .*;$',
            after=stanza,
            use_sudo=True,
        )


@traced
def nginx_limit_workers(count='auto'):
    from fabric.contrib.files import sed, contains, append

    filename = '/etc/nginx/nginx.conf'
    stanza = 'worker_processes %s;' % (count,)
    if not exists(filename):
        return
    if contains(filename, 'worker_processes'):
        sed(
            filename,
            before='^worker_processes.*;$',
            after=stanza,
            use_sudo=True,
        )
    else:
        append(
            filename,
            stanza,
            use_sudo=True,
        )


@contextmanager
def vagrant_ssh(box=None, port=15022):
    """Context manager for operations to run as vagrant ssh user"""
    # change from the default user to 'vagrant'
    box = box or ''
    result = local(
        'vagrant ssh-config %(box)s | grep IdentityFile' % locals(), capture=True
    )
    key_filename = result.split()[1].strip('"')
    print("Keyfile: %s" % (key_filename,))
    with settings(
        user='vagrant',
        hosts=['127.0.0.1:%s' % (port,)],
        key_filename=key_filename,
    ):
        yield
        # use vagrant ssh key


def create_product_user_vagrant(port=None, box=None):
    """Create vagrant hosted build machine"""
    if port is None:
        port = env.port
    # TODO: this might be more useful if it defaulted
    # to the product name, as that's what the vagrant
    # files use...
    box = box or env.product or ''
    local('vagrant up %(box)s' % locals())
    user = env.user
    with vagrant_ssh(box, port):
        create_product_user(user, 'vagrant')


def versative_licenses():
    """Generate versative (xfactor) licenses (requires /var/epgdata/arcos.lmx secret key)"""
    mac = run('cat /sys/class/net/eth0/address').strip()
    license_files = (
        local('versalic-autolicense %(mac)s' % locals(), capture=True)
        .strip()
        .splitlines()
    )
    if not len(license_files) == 2:
        raise RuntimeError("Did not create the versative licenses")
    for license in license_files:
        if exists('/var/chroot/fsroot'):
            target = os.path.join('/var/chroot/fsroot')
        else:
            target = '/'
        put(
            license,
            os.path.join(
                target,
                'var/firmware/arcos/3screens/license/',
                os.path.basename(license),
            ),
        )


@contextmanager
def ceton_ucrypt():
    ucrypt_key_paths = [
        '~/.ssh/ucrypt.private',
        os.path.join(
            os.path.dirname(__file__),
            '..',
            '..',
            '..',
            'signing-keys',
            'ucrypt.private',
        ),
    ]
    ssh_key = None
    for possible in ucrypt_key_paths:
        if os.path.exists(possible):
            ssh_key = possible
            os.chmod(possible, 0o600)
    if not ssh_key:
        raise RuntimeError(
            "Need the ucrypt key in one of:\n  %s" % ("\n  ".join(ucrypt_key_paths),)
        )
    with settings(
        user='ceton',
        host_string='%s:2222' % (env.host,),
        key_filename=ssh_key,
    ):
        yield None


@traced
def fix_locales():
    if hasattr(env.target_os, 'ensure_locales'):
        env.target_os.ensure_locales()
    env.target_os.system_encoding()


@parallel
def analog_status():
    run('~/Arcos3ScreensApp/AnalogUtility/AnalogUtility -status')


def debug_machine(disable=False):
    """Configure the machine to run in (unsafe) debug mode"""
    content = '''[django]
debug=True
session_duration=604800
'''
    filename = '/etc/firmware/django-debug.conf'
    if disable:
        sudo('rm %(filename)s' % locals())
    else:
        env.target_os.write_string_to_file(content, filename)


@traced
def ensure_python(target_python=TARGET_PYTHON):
    """Ensure that target python is present/available

    Centos7 only currently
    """
    if env.target_os.name == 'centos7':
        if target_python != 'python2.7' and not exists('/usr/bin/python3.6'):
            bc = checkout_binary_components()
            local_source = os.path.join(bc, 'python3')
            target = '/tmp/python3'
            sudo('mkdir -p target')
            project.rsync_project(
                target,
                local_source + '/*',
            )
            sudo('mkdir -p /src/binary-components/python3')
            sudo('rsync /tmp/python3/* /src/binary-components/python3/')
            env.target_os.remove_python27()
            sudo('rm -rf /tmp/python3')


@traced
def selinux_compile(root=SOURCE_ROOT):
    """Compile an selinux te_file on the host machine (needs policycoreutils-python on centos7)"""
    for path, subdirs, files in os.walk(SOURCE_ROOT):
        for file in files:
            if file.endswith('.te'):
                te_file = os.path.join(path, file)
                working = put(te_file, '~/')
                basename = os.path.splitext(os.path.basename(te_file))[0]
                sudo('checkmodule -M -m -o %(basename)s.mod %(basename)s.te' % locals())
                sudo(
                    'semodule_package -o %(basename)s.pp -m %(basename)s.mod' % locals()
                )
                get('~/%(basename)s.pp' % locals(), path)


@traced
def pack_rpm_update(name, *packages):
    """Create an RPM update that pulls given packages"""
    working = os.path.join('/tmp/package-download-rpms')
    if exists(working):
        sudo('rm -rf %(working)s' % locals())
    sudo('mkdir -p %(working)s' % locals())
    venv = env.virtual_env
    with cd(working):
        packages = ' '.join(['%s' % (package) for package in packages])
        sudo('yum install --downloadonly --downloaddir=./ %(packages)s' % locals())
        sudo('%(venv)s/bin/rpm-metadata -n %(name)s -d ./' % locals())
        get('%(working)s/*' % locals(), SIDELOADS + '/')
    sudo('rm -rf %(working)s' % locals())


def test_os_updates():
    """Run OS updates on the host with currently configured sideloads

    * do a local sideload registry upload
    * update osupdates on the target (note, if your update isn't in osupdates
      then you'll need to forceinstall:package on the package where the updates
      are (likely your product package))
    * embed sideloads
    * run os-updates with `--no-network` flag
    """
    local('sideload-rebuild')
    embed_sideloads()
    forceinstall('osupdates')
    sudo(
        '%(env)s/bin/os-updates --no-network'
        % {
            'env': env.virtual_env,
        }
    )


def approximate_date():
    """Set approximate date so the machine can pull packages etc..."""
    format = '%Y%m%d'
    current = time.strftime(format)
    sudo("date +%(format)s -s '%(current)s'" % locals())


def docker_commit():
    HERE = os.path.dirname(__file__)
    local('./docker-dev/start-dev.py --commit')


@traced
def ipv4_dns_preference():
    """Switch to the use of ipv4 dns rather than ipv6 to avoid issues on ipv6-unrouteable networks"""
    sudo(
        "sed -i -- 's/#precedence ::ffff:0:0\/96  100/precedence ::ffff:0:0\/96  100/g' /etc/gai.conf"
    )


def pypi_push(packages='/tmp/packages/*.whl', user='mcfletch'):
    """Use twine to push our packages to the atx pypi server

    Normally you'd do this after running a release to ensure that
    the packages we used to build the release are available to
    future builds without needing to recompile.
    """
    environment = env.virtual_env
    sudo('%(environment)s/bin/pip install twine' % locals())
    run(
        '%(environment)s/bin/twine upload --repository-url http://10.1.0.213:8080 -u %(user)s --skip-existing %(packages)s'
        % locals()
    )


def remote_architecture():
    return as_unicode(run('uname -m')).strip().splitlines()[-1]

    del HERE


def pods_by_label(key, value):
    """Find all pods with labels where key == value"""
    pods = json.loads(nbio.Process('kubectl get pods -o json')())['items']
    result = []
    for pod in pods:
        metadata = pod.get('metadata', {})
        labels = metadata.get('labels')
        if labels:
            pod_value = labels.get(key)
            if pod_value == value:
                result.append(pod)
    return result


def pods_with_hostname(hostname):
    """Return pods which match hostname"""
    pods = json.loads(nbio.Process('kubectl get pods -o json')())['items']
    result = []
    for pod in pods:
        pod_host = pod['spec'].get('hostname')
        if pod_host == hostname:
            log.info("Match %s", pod_host)
            result.append(pod)
        log.info('Pod: %s', pod_host)
    return result


def k8s_run_on_container(command, container):
    log.info("Running on %s", container)
    return nbio.Process(['kubectl', 'exec', '-t', container, '--', command])()
