import os
import django.utils
import six

__all__ = [
    'EMAIL_BACKEND',
    'INSTALLED_APPS',
    'LOGGING',
    'MIDDLEWARE',
    'PASSWORD_HASHERS',
    'PROTECTED_URL',
    'STATICFILES_FINDERS',
    'TEMPLATES',
    'WEBPACK_LOADER',
    'DEFAULT_AUTO_FIELD',
]

if not hasattr(django.utils, 'six'):
    django.utils.six = six

EMAIL_BACKEND = 'sysemail.models.EmailBackend'
HERE = os.path.dirname(__file__)
DEFAULT_AUTO_FIELD = 'django.db.models.BigAutoField'


def get_external_ip(default=None):
    import netifaces

    for iface in sorted(netifaces.interfaces()):
        if iface == 'lo':
            continue
        for address in netifaces.ifaddresses(iface).get(netifaces.AF_INET, []):
            if address.get('addr'):
                return address.get('addr')
    return default


STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'django_assets.finders.AssetsFinder',
]
TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                # Insert your TEMPLATE_CONTEXT_PROCESSORS here or use this
                # list if you haven't customized them:
                'django.contrib.auth.context_processors.auth',
                'django.template.context_processors.debug',
                'django.template.context_processors.i18n',
                'django.template.context_processors.media',
                'django.template.context_processors.static',
                'django.template.context_processors.tz',
                'django.template.context_processors.request',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

MIDDLEWARE = [
    'atxstyle.middleware.SetRemoteAddrFromForwardedFor',
    'atxstyle.middleware.SettingsMiddleWare',
    'atxstyle.middleware.ErrorLogMiddleWare',
    'django.middleware.common.CommonMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'atxstyle.middleware.SystemMiddleWare',
]

INSTALLED_APPS = [
    'django.contrib.contenttypes',
    'django.contrib.auth',
    'django.contrib.sessions',
    'django.contrib.sites',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'netconfig',
    'atxstyle',
    'sysemail',
    'django_assets',
    'webpack_loader',
    'reactforms',
    'restore',
]

PASSWORD_HASHERS = [
    'django.contrib.auth.hashers.BCryptSHA256PasswordHasher',
    'django.contrib.auth.hashers.BCryptPasswordHasher',
    'django.contrib.auth.hashers.PBKDF2PasswordHasher',
    'django.contrib.auth.hashers.PBKDF2SHA1PasswordHasher',
    'django.contrib.auth.hashers.SHA1PasswordHasher',
    'django.contrib.auth.hashers.MD5PasswordHasher',
    'django.contrib.auth.hashers.CryptPasswordHasher',
]

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'syslog': {
            'format': '%(name)s: <%(levelname)s> %(message)s',
        },
        'auth': {
            'class': 'logging.Formatter',
            'format': '%(created)s|%(levelname)s|%(message)s',
        },
        'full': {
            'class': 'logging.Formatter',
            'format': '%(levelname)- 7s %(name)-30s %(asctime)s  %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
        },
    },
    'handlers': {
        'json': {
            'level': 'INFO',
            'class': 'fussy.jsonlog.JSONSocketHandler',
            'host': os.environ.get('JSON_LOGGING_HOST', 'localhost'),
        },
        'mail_admins': {
            'level': 'ERROR',
            'class': 'django.utils.log.AdminEmailHandler',
            'filters': [],
        },
        'console': {
            'level': 'DEBUG',
            'class': 'logging.StreamHandler',
            'formatter': 'full',
        },
        'auth': {
            'level': 'INFO',
            'class': 'logging.handlers.RotatingFileHandler',
            'maxBytes': 2 * 1024 * 1024,
            'backupCount': 1,
            'formatter': 'auth',
        },
    },
    'loggers': {
        '': {
            'handlers': ['console', 'json'],
            'level': 'DEBUG',
        },
        'audit': {
            'handlers': ['json'],
            'level': 'INFO',
        },
        'django.db.backends': {
            'handlers': ['console'],
            'level': 'INFO',
            'propagate': False,
        },
        'django.request': {
            'handlers': ['json'],
            'level': 'ERROR',
            'propagate': True,
        },
        'django.auth': {
            'handlers': ['auth'],
            'level': 'INFO',
            'propagate': False,
        },
        'django.server': {
            'handlers': ['mail_admins'],
            'level': 'ERROR',
            'propagate': False,
        },
        'django.security.DisallowedHost': {
            'level': 'ERROR',
            'handlers': ['console'],
            'propagate': False,
        },
    },
}

if os.path.exists('/dev/log'):
    LOGGING['handlers']['syslog'] = {
        'level': 'WARN',
        'class': 'logging.handlers.SysLogHandler',
        'address': '/dev/log',
        'formatter': 'syslog',
    }
    LOGGING['loggers']['']['handlers'].append('syslog')


if not os.environ.get('WEBWORKER'):
    # log.warning( 'Disabling auth logging for root use' )
    for key in ['formatters', 'loggers', 'handlers']:
        for to_delete in ('auth', 'django.auth'):
            try:
                del LOGGING[key][to_delete]
            except KeyError as err:
                pass
else:
    log_filename = os.path.join('/var', 'firmware', 'log', 'django.log')
    if os.path.exists(os.path.dirname(log_filename)):
        LOGGING['handlers']['firmware'] = {
            'level': 'DEBUG',
            'class': 'logging.handlers.RotatingFileHandler',
            'maxBytes': 2 * 1024 * 1024,
            'backupCount': 10,
            'filename': log_filename,
            'formatter': 'full',
        }
        LOGGING['loggers']['']['handlers'].append('firmware')

if os.environ.get('DEBUG_LOG'):
    LOGGING['handlers']['debug'] = {
        'level': 'DEBUG',
        'class': 'logging.handlers.RotatingFileHandler',
        'filename': os.environ.get('DEBUG_LOG'),
        'maxBytes': 2 * 1024 * 1024,
        'backupCount': 1,
        'formatter': 'full',
    }
    LOGGING['loggers']['']['handlers'].append('debug')

PROTECTED_URL = '/protected/'

WEBPACK_LOADER = {
    'DEFAULT': {
        'BUNDLE_DIR_NAME': 'bundles/',
        'STATS_FILE': (
            os.path.join(HERE, 'static/bundles/webpack-stats.json')
            if not os.environ.get('WEBWORKER')
            else '/opt/firmware/current/www/static/bundles/webpack-stats.json'
        ),
    }
}


def remove_assets(setting):
    """Remove webassets from the product's settings"""
    SUPPRESSED_APPS = set(
        [
            'django_assets',
            'django_assets.finders.AssetsFinder',
        ]
    )
    return [x for x in setting if x not in SUPPRESSED_APPS]


def remove_webpack(setting):
    """Remove webpack from pre-webpack applications"""
    SUPPRESSED_APPS = set(
        [
            'webpack_loader',
        ]
    )
    return [x for x in setting if x not in SUPPRESSED_APPS]


if 'DJANGO_EXTENSIONS' in os.environ:
    INSTALLED_APPS.append('django_extensions')
