"""Cache helpers for using low-level caches in the GUI"""
import logging
from functools import wraps
from django.conf import settings

log = logging.getLogger(__name__ + '.cache')


def cached_by_key(suffix, cache):
    """Cache per-instance results by js_key(instance)-suffix in given cache"""

    def with_cached_lookup(function):
        from .models import js_key

        @wraps(function)
        def using_cache(self, *args, **named):
            key = cache.make_key('%s-%s' % (js_key(self), suffix))
            result = cache.get(key)
            if result is not None:
                return result
            if settings.DEBUG:
                log.info("Recalculating cache for key %s", key)
            result = function(self, *args, **named)
            cache.set(key, result)
            return result

        return using_cache

    return with_cached_lookup


def clear_cache(cache):
    """Clear the cache without wiping out all redis keys"""
    cache.delete_pattern('*')
