"""Generate a licensing keyring for a cluster-hosted namespace"""
import json, uuid, tempfile, shutil, os, hashlib, base64
from fussy import twrite, nbio
from atxstyle import encryption, uniquekey
from pyrsistent import optional

KEY_SOURCE = os.path.join(
    os.path.dirname(os.path.abspath(encryption.__file__)),
    '../common-config/etc/atxlicense/keys',
)


def get_namespace_uid(namespace):
    """Gets the UID of the namespace in the current cluster"""
    content = (
        nbio.Process(
            "kubectl get namespace/%(namespace)s -o jsonpath='{.metadata.uid}'"
            % locals()
        )()
        .decode('utf-8')
        .strip()
    )
    if not content:
        raise RuntimeError('Expected a UUID from the namespace')
    print('UUID content: %s' % (content,))
    return uuid.UUID(content)


def write_unique_key(filename, uid):
    data = {
        'original': str(uid),
        'baseboard': str(uid),
        'system': str(uid),
        'baseboard_product': 'kubernetes',
        'baseboard_version': None,
        'bios_version': None,
        'processor_version': None,
    }
    twrite.twrite(filename, json.dumps(data))


def armour(value):
    if not isinstance(value, bytes):
        value = value.encode('utf-8')
    value = base64.encodebytes(value)
    value = value.replace(b'\n', b'').replace(b'\r', b'')
    return value.decode('utf-8')


def generate_gpg_key(namespace):
    uid = get_namespace_uid(namespace)
    workdir = tempfile.mkdtemp(prefix='gpg-', suffix='-%(namespace)s' % locals())
    result = {}
    try:
        uniquekey.KEY_STORE = os.path.join(workdir, 'uniquekey.json')
        encryption.ETC_DIR = workdir
        encryption.GLOBAL_KEYS = os.path.join(workdir, 'keys')
        shutil.copytree(KEY_SOURCE, encryption.GLOBAL_KEYS)
        encryption.ensure_unique_key = lambda: None
        env = os.environ.copy()
        env['GNUPGHOME'] = encryption.GLOBAL_KEYS

        write_unique_key(uniquekey.KEY_STORE, uid)
        assert uniquekey.get_base_key() == str(uid), uniquekey.get_base_key()
        encryption.create_gpg_cert(encryption.GLOBAL_KEYS, ownership='')
        result['uniquekey.json'] = armour(open(uniquekey.KEY_STORE, 'rb').read())
        key = nbio.Process(
            [
                'gpg',
                '--export-secret-keys',
                '-a',
                str(uid),
            ],
            env=env,
        )()
        if not key.strip():
            raise ValueError("Did not get the expected key in ascii format")
        result['private-key.txt'] = armour(key)
        result['local-key-identity'] = armour(
            open(os.path.join(workdir, 'local-key-identity'), 'rb').read()
        )
        return result
    finally:
        # print('Delete %s' % (workdir,))
        shutil.rmtree(workdir)


def create_secret(namespace, gpg_key):
    """Encode the secret files for use in final target..."""
    template = """### Generated by clusteruniquekey
apiVersion: v1
kind: Secret
metadata:
    name: atx-license-keys
    namespace: %(namespace)s
type: Opaque
data:
    %(keys)s
"""
    keys = "\n    ".join(['    %s: %s' % (k, v) for (k, v) in gpg_key.items()])
    return template % locals()


def get_options():
    import argparse

    parser = argparse.ArgumentParser(
        description='Create a licensing key-ring secret for a specific instance of a cluster namespace in k8s'
    )
    parser.add_argument(
        'namespace', help='Namespace in the cluster which will be licensed'
    )
    parser.add_argument(
        'filename',
        help='Filename into which to write the file',
        default='-',
    )
    return parser


def main():
    options = get_options().parse_args()
    secret = create_secret(options.namespace, generate_gpg_key(options.namespace))
    if options.filename not in ('', '-'):
        twrite.twrite(options.filename, secret)
    else:
        print(secret)
