"""Some basic threading/control primitives for our communications pleasure"""
import threading,logging,os,time
from functools import wraps
from fussy.processcontrol import *
from fussy import filewatch, twrite
log = logging.getLogger( __name__ )

DEBUG = False
if not DEBUG:
    log.setLevel( logging.WARN )
    

def with_frequency_filter( key,  period=3600 ):
    """Create a wrapper that will only run the function at most once per period
    
    Uses a file's stat to determine the frequency of messages
    """
    directory = '/var/firmware/run'
    filename = os.path.join( directory, key + '.last' )
    def last_time():
        try:
            last = os.stat( filename ).st_mtime
        except Exception:
            last = 0
        return last
    def frequency_filter_dec( function ):
        @wraps(function)
        def frequency_filter( *args,  **named ):
            force = False
            if 'force_reset' in named:
                named.pop('force_reset')
                force = True
            if (force or time.time() - last_time() > period):
                twrite.twrite( filename,  '' )
                return function( *args,  **named )
            else:
                log.debug('Skipping run of %s', function)
                return None
        return frequency_filter 
    return frequency_filter_dec


PIPES_TO_KILL = []
def register_pipe( pipe ):
    """Register pipe for exit cleanups/killing"""
    PIPES_TO_KILL.append( pipe )
def close_pipe( pipe ):
    """Close a given pipe"""
    if pipe.returncode is None:
        rc = pipe.poll()
        if rc is None:
            kill_kill_kill( pipe.pid )
    while pipe in PIPES_TO_KILL:
        try:
            PIPES_TO_KILL.remove( pipe )
        except Exception:
            # race condition possible
            pass
def cleanup_pipes( ):
    """Close/cleanup all registered pipes"""
    while PIPES_TO_KILL:
        try:
            pipe = PIPES_TO_KILL.pop()
        except Exception:
            # race condition possible
            pass
        else:
            close_pipe( pipe )

class FileWatcher( object ):
    """Watch for updates to the given directory/path
    
    Filters out the fussy.filewatch.FileWatcher events to 
    only include non-directory, existing filenames
    """
    def __init__( self, directory, filenames='*[!~]' ):
        '''Initialize the file watcher
        
        directory -- directory in which to watcher
        filenames -- glob/fnmatch style specification of files to watch
        '''
        self._fw = filewatch.FileWatcher( directory, filenames )
    def __iter__( self ):
        log.info( 'Setting up inotify operation on %s/%s', self._fw.directories[0],self._fw.pattern )
        self.active = True
        for event in self._fw:
            if event.existing and not event.isdir:
                yield event.full_name

def shovel( filename, queue ):
    """Shovel the contents of filename into queue on every change
    
    Sends initial contents of filename before beginning...
    """
    directory = os.path.dirname( filename )
    if not os.path.exists(directory):
        os.makedirs(directory)
    watcher = FileWatcher( directory, os.path.basename(filename) )
    for fullpath in watcher:
        log.info( 'Reloading: %s', fullpath )
        try:
            queue.put( open( fullpath ).read() )
        except Exception as err:
            # race condition, already gone before we read it...
            log.warning( 'Unable to reload from %s: %s', fullpath, err )

class WithLock( object ):
    def __init__( self, blocking=True ):
        self.lock = threading.RLock()
        self.blocking = blocking 
    def __enter__( self ):
        if self.lock.acquire( self.blocking ):
            return True 
        raise RuntimeError( """Unable to immediately acquire lock""" )
    def __exit__( self, *args ):
        self.lock.release()
    def __call__( self, function ):
        def wrap_with_lock( *args, **named ):
            with self:
                return function(*args, **named )
        wrap_with_lock.__name__ = function.__name__ 
        wrap_with_lock.__doc__ = function.__doc__
        return wrap_with_lock 
