from django.http import (
    Http404,
    HttpResponse,
    HttpResponseForbidden,
    StreamingHttpResponse,
    HttpResponseRedirect,
)
from django.views.decorators.csrf import csrf_exempt
from django.core.validators import ValidationError
from atxstyle.sixish import unicode
from atxstyle import utctime
import datetime
from annoying.decorators import render_to

assert render_to
from django.contrib.auth.decorators import (
    login_required as base_login_required,
    permission_required as base_permission_required,
)
from django.core.exceptions import PermissionDenied
from django.conf import settings
from django.db.models.fields.files import FieldFile
from functools import wraps
from glob import glob
from django.urls import reverse
import contextlib
import logging, os, random
import json
from . import jsonencode
from fussy import twrite
from .protectedstorage import with_protected_redirect, protected_redirect
from .cachehelpers import cached_by_key

log = logging.getLogger(__name__)
assert with_protected_redirect

CLUSTER_LOCAL_MARKER = os.environ.get('CLUSTER_LOCAL_HOST_MARKER', 'MOO')


def is_cluster_local(request):
    if CLUSTER_LOCAL_MARKER:
        provided = request.META.get('HTTP_CLUSTER_LOCAL_HOST_MARKER')
        if provided == CLUSTER_LOCAL_MARKER:
            return True
    return False


def is_local(request):
    if request.META.get('REMOTE_ADDR') in _get_local_ips(request):
        return True
    if is_cluster_local(request):
        return True
    return False


def is_ajax(request):
    """Django migration where is_ajax is deprecated in 3.1"""
    if hasattr(request, 'accepts'):
        # modern Django
        return request.accepts("application/json")
    else:
        return request.is_ajax()


def _get_local_ips(request):
    return ['localhost', '127.0.0.1']


def login_required(function):
    base_wrap = base_login_required(function)

    @wraps(function)
    def wrapper(request, *args, **named):
        if is_ajax(request) and not request.user.is_authenticated:
            return HttpResponse(
                '{"error":true,"messages":["Unauthenticated Ajax Request"]}',
                content_type='application/json',
                status=401,
            )
        return base_wrap(request, *args, **named)

    return wrapper


def permission_required(perm, *args, **named):
    base_perm = base_permission_required(perm, *args, **named)

    def decorator(function):
        base_wrap = base_perm(function)

        @wraps(function)
        def wrapper(request, *args, **named):
            if is_ajax(request) and not request.user.has_perm(perm):
                return HttpResponse(
                    '{"error":true,"messages":["Authenticated Ajax Request Lacks Required Permission"]}',
                    content_type='application/json',
                    status=401,
                )
            return base_wrap(request, *args, **named)

        return wrapper

    return decorator


def with_section(section):
    def wrap_with_section(function):
        @wraps(function)
        def section_dec(request, *args, **named):
            request.section = section
            request.menu = [
                m
                for m in settings.MENU
                if (
                    (not m)
                    or (not m.get('permission'))
                    or (request.user and request.user.has_perm(m['permission']))
                )
            ]
            found = False
            for m in request.menu:
                if m and m.get('section') == section:
                    found = True
            if request.menu and not found:
                request.section = request.menu[0].get('section')
            return function(request, *args, **named)

        return section_dec

    return wrap_with_section


def render_to_json(function):
    """Produce json output from given function"""

    @wraps(function)
    def as_json(*args, **named):
        try:
            result = function(*args, **named)
            if isinstance(result, HttpResponseRedirect) and result.url.startswith(
                reverse('login')
            ):
                raise PermissionDenied('Permission denied')
        except Http404 as err:
            log.info("404: %s", err)
            return HttpResponse(
                jsonencode.dumps(
                    {
                        'error': True,
                        'messages': [str(err)],
                    }
                ),
                content_type='application/json',
                status=404,
            )

        except PermissionDenied as err:
            if (
                args
                and hasattr(args[0], 'is_authenticated')
                and args[0].is_authenticated
            ):
                # We don't want to log every logged-out request for a page...
                log.warning(
                    'Permission denied on %s with %s',
                    function.__name__
                    if function.__name__ != 'react_form'
                    else (
                        function.__self__ if hasattr(function, '__self__') else function
                    ),
                    err,
                )
            return HttpResponse(
                jsonencode.dumps(
                    {
                        'error': True,
                        'auth_failure': True,
                        'messages': [err.args[0]],
                    }
                ),
                content_type='application/json',
            )
        except Exception as err:
            log.exception('Error during json view: %s', err)
            formatted = jsonencode.dumps({"error": True, "messages": [unicode(err)]})
            return HttpResponse(formatted, content_type='application/json')
        if isinstance(result, (bytes, unicode)):
            return HttpResponse(result, content_type='application/json')
        elif isinstance(result, (HttpResponse, StreamingHttpResponse)):
            return result
        else:
            try:
                formatted = jsonencode.dumps(result)
            except Exception as err:
                log.exception('Error during json dump: %s', result)
                formatted = jsonencode.dumps({"error": True, "message": [unicode(err)]})
            return HttpResponse(formatted, content_type='application/json')

    return as_json


def errors_to_json(function):
    """Decorate the function so that errors produce json-api-compatible reports"""
    from django.urls import reverse

    @wraps(function)
    @render_to_json
    def with_error_conversion(request, *args, **named):
        try:
            response = function(request, *args, **named)
            if isinstance(response, HttpResponseForbidden):
                raise PermissionDenied()
            elif isinstance(response, HttpResponseRedirect):
                if response.url.startswith(reverse('login')):
                    # function(request, *args, **named)
                    raise PermissionDenied('Need Login')
            return response
        except PermissionDenied as err:
            log.info("Returning permission denied on %s: %s", request.path, err)
            return {
                'error': True,
                'auth_failure': True,
                'messages': [
                    err.args[0],
                ],
                'next': request.path,
            }
        except Exception as err:
            return {
                'error': True,
                'messages': [str(err)],
            }

    return with_error_conversion


def local_or_login_required(function):
    """Require that the request *either* be logged in *or* be from localhost"""
    protected = login_required(function)

    @wraps(function)
    def with_local_or_login(request, *args, **named):
        if is_local(request):
            # TODO: allow for IPv6 local or local-hostname..
            return function(request, *args, **named)
        else:
            return protected(request, *args, **named)

    return with_local_or_login


def local_required(function):
    """Require a localhost connection"""

    @wraps(function)
    def with_local(request, *args, **named):
        if is_local(request):
            # TODO: allow for IPv6 local or local-hostname..
            return function(request, *args, **named)
        return HttpResponseForbidden('Only local users permitted')

    return with_local


def log_login_failures(function):
    """Logs any post to the page that doesn't return a 302 (redirect) as a login failure

    Logged into the logger at 'django.auth' to allow for configs to write the
    logs separately...
    """
    auth_log = logging.getLogger('django.auth')

    @wraps(function)
    def wrapped(request, *args, **named):
        result = function(request, *args, **named)
        if request.method == 'POST' and result.status_code != 302:
            auth_log.error(
                "Login Fail %r by %s",
                request.POST.get('username'),
                request.META.get('REMOTE_ADDR'),
            )
        return result

    return wrapped


def reversion_unchanged(function):
    from reversion import models as reversion_models
    from fussy import twrite
    from atxstyle import utctime
    from django.http import HttpResponseNotModified
    from django.core.cache import cache
    from django.conf import settings

    cache_key = 'reversion-unchanged-%s.%s' % (function.__module__, function.__name__)
    cache_dir = os.path.join(settings.PROTECTED_DIR, 'reversion-unchanged')
    if not os.path.exists(cache_dir):
        os.makedirs(cache_dir, 0o755)
    cache_file = os.path.join(cache_dir, cache_key)

    @wraps(function)
    def with_reversion_check(request, *args, **named):
        if request.method == 'GET' and 'ts' in request.GET:
            # we can short-circuit the request if we haven't updated the config
            # since ts...
            try:
                dt = utctime.from_timestamp(float(request.GET['ts']))
            except (TypeError, ValueError):
                log.exception("Unable to convert %r to a timestamp", request.GET['ts'])
            else:
                # note the offset here to avoid icky corner cases
                if not reversion_models.Revision.objects.filter(
                    date_created__gt=dt
                ).count():
                    return HttpResponseNotModified()
        if request.method == 'GET':
            current = cache.get(cache_key)
            if current:
                fresh = False
                try:
                    ts, content_type = json.loads(current)
                except Exception:
                    log.exception("Unable to load cached structure")
                else:
                    ds = utctime.from_timestamp(ts)
                    if (
                        ts
                        and not reversion_models.Revision.objects.filter(
                            date_created__gt=ds
                        ).count()
                    ):
                        fresh = True
                    if fresh:
                        return protected_redirect(
                            cache_file,
                            request,
                            content_type=content_type,
                        )
        last_revision = reversion_models.Revision.objects.order_by('-date_created')[
            :1
        ].first()
        response = function(request, *args, **named)
        if last_revision:
            cache.set(
                cache_key,
                json.dumps(
                    [
                        utctime.as_timestamp(last_revision.date_created),
                        response['Content-Type'],
                    ]
                ),
            )
            twrite.twrite(cache_file, response.content)
        return response

    return with_reversion_check


@contextlib.contextmanager
def with_reversion_record(user_id=None, comment=None):
    """Run the statements creating a reversion record when complete"""
    from atxstyle import utctime
    from reversion import models as reversion_models

    try:
        yield
    except Exception:
        log.info("Skipping revision recording")
        raise
    else:
        reversion_models.Revision.objects.create(
            date_created=utctime.current_utc(),
            user_id=user_id,
            comment=comment or 'User Edit',
        )


def ajax_error(request, message, code=401):
    if is_ajax(request):
        return HttpResponse(
            jsonencode.dumps(
                {
                    'error': True,
                    'messages': [message],
                }
            ),
            status=code,
        )
    return HttpResponseForbidden(message)


def with_ajax_login(function):
    """Allow post of credentials via header-based authentication"""

    @wraps(function)
    def handle_ajax_login(request, *args, **named):
        if not request.user.is_authenticated:
            auth = request.META.get('HTTP_X_UCRYPT_AUTH')
            if not auth:
                return ajax_error(
                    request, 'Credentials not Provided in X-UCRYPT-AUTH', 401
                )
            request.META['HTTP_X_REQUESTED_WITH'] = 'XMLHttpRequest'
            try:
                username, password = auth.split(':', 1)
            except ValueError:
                return ajax_error(
                    request,
                    'Credentials Incorrectly Formatted "username:passphrase" expected',
                    401,
                )
            from django.contrib.auth import authenticate, login

            user = authenticate(username=username, password=password)
            if user is not None:
                login(request, user)
            else:
                return ajax_error(request, 'Credentials Incorrect', 403)
        return function(request, *args, **named)

    return handle_ajax_login


def cprofile(filename):
    """Decorator to cProfile to the given filename for the function"""
    import cProfile

    profiler = cProfile.Profile()

    def cprofile_wrapper(function):
        @wraps(function)
        def with_wrapper(*args, **named):
            profiler.enable()
            try:
                return function(*args, **named)
            finally:
                profiler.disable()
                profiler.dump_stats(filename)

        return with_wrapper

    return cprofile_wrapper


def force_to_choices(clean_function):
    from functools import wraps

    assert clean_function.__name__.startswith('clean_')
    field = clean_function.__name__[6:]

    def convert_choice(self, base):
        for k, v in self.fields[field].choices:
            if k == 'null':
                k = None
            if unicode(base).lower() == unicode(k).lower():
                return k
        raise ValidationError("%r is not recognized as a selection")

    @wraps(clean_function)
    def wrapper(self):
        base = self.cleaned_data.get(field)
        if isinstance(base, (tuple, list)):
            base = [convert_choice(self, b) for b in base]
        else:
            base = convert_choice(self, base)
        return clean_function(self, base)

    return wrapper


def with_basic_auth():
    """Require basic auth, already-logged-in, or local connection"""
    from rest_framework.authentication import BasicAuthentication
    from django.contrib.auth import login

    def with_auth(function):
        @wraps(function)
        def checking(request, *args, **named):
            if not (request.user.is_authenticated or is_local(request)):
                user_auth_tuple = BasicAuthentication().authenticate(request)
                if user_auth_tuple:
                    login(request, user_auth_tuple[0])
                else:
                    log.warning(
                        'Authentication failure for login with %s',
                        request,
                    )
                    return ajax_error(request, 'Authentication failed', 401)
            return function(request, *args, **named)

        return checking

    return with_auth


def with_modified_tracker(key, watches=(), prefix='if-mod-since-'):
    """Record timestamp if we have no cache for a key, record time of last modification for watches

    creates DB signal watches for each model-class in watches,
    when there is a save of one of them, the last-update is saved
    into the cache value and the if-modified-since checks will
    now return fail if they are for the older times...
    """
    from django.core import cache
    from django.db import models
    from django.views.decorators.http import condition

    final_key = '%s-%s' % (
        prefix,
        key,
    )

    def on_change(**args):
        dt = utctime.current_utc()
        cache.cache.set(final_key, dt)
        return dt

    def check_last(request, *args, **named):
        last = cache.cache.get(final_key)
        if last is None:
            last = on_change()
        # log.info(
        #     'Request If-Modified-Since: %s', request.META.get('HTTP_IF_MODIFIED_SINCE')
        # )
        # log.info('Reporting last modified for %s of %s', final_key, last)
        return last

    for watch in watches:
        models.signals.post_save.connect(on_change, sender=watch)
        models.signals.post_delete.connect(on_change, sender=watch)

    def modified_tracker_wrapper(function):
        @condition(last_modified_func=check_last)
        @wraps(function)
        def with_modified_tracker(request, *args, **named):
            return function(request, *args, **named)

        return with_modified_tracker

    return modified_tracker_wrapper
