"""Setup product-default settings environment variable"""
import sys, os, getpass, glob
from six.moves import configparser
ALREADY_SETUP = False

SETTINGS_KEY = 'DJANGO_SETTINGS_MODULE'

def get_product( product=None ):
    """Determine the default product for this installation"""
    if product is None:
        if SETTINGS_KEY in os.environ:
            product = os.environ[SETTINGS_KEY].split('.')[0]
        elif os.path.lexists('/opt/firmware') and os.path.islink('/opt/firmware'):
            product = os.path.basename(os.readlink('/opt/firmware'))
        elif os.path.isdir('/opt/firmware'):
            # Reverse links, find the link from name => firmware (should only ever be passport)
            for link in glob.glob('/opt/*'):
                if os.path.islink(link) and os.readlink(link).endswith('firmware'):
                    product = os.path.basename(link)
        elif getpass.getuser() != 'root':
            product = getpass.getuser()
        elif 'SUDO_USER' in os.environ:
            product = os.environ['SUDO_USER']
        elif os.path.exists('/etc/firmware/release.conf'):
            parser = configparser.SafeConfigParser()
            parser.read('/etc/firmware/release.conf')
            product = parser.get('release','product')
        else:
            for path in glob.glob('/home/*/product-user.txt'):
                base = os.path.basename(os.path.dirname(path))
                if base == 'recovery':
                    continue
                # On EPGData this *could* find 'digistream' instead of epgdata...
                product = base
            raise RuntimeError("No product specified, no generic symlink, no username we can use")
    if not product:
        raise RuntimeError("Need a product to load settings, got: %r"%(product,))
    if not SETTINGS_KEY in os.environ:
        if product == 'digistream':
            settings = 'digiconfig.settings'
        else:
            settings = '%s.settings'%(product,)
    else:
        settings = os.environ[SETTINGS_KEY]
    return product,settings

def get_user(product):
    product,settings = get_product(product)
    return product if product != 'egpdata' else 'digistream'

def get_home(product=None):
    user = get_user(product)
    return '/home/%(user)s'%locals()

def restart_as_user(product=None):
    """Given a product if we are root, run as product user"""
    if os.geteuid() == 0:
        user = get_user(product)
        command = ['sudo','-n','-u',user]+sys.argv
        return os.execvp('sudo',command)


def setup_default(product=None, dev=False):
    """Makes sure that IFF you haven't got a settings we use product-default

    Also does django.setup() call...
    """
    global ALREADY_SETUP
    ALREADY_SETUP = ALREADY_SETUP or bool(sys.modules.get('django'))
    if ALREADY_SETUP:
        return
    ALREADY_SETUP = True
    product,settings = get_product( product )
    os.environ[SETTINGS_KEY] = settings
    package,module = settings.split('.')
    if module != 'settings':
        raise RuntimeError("Seems like an attempt to work around us?")
    elif not os.path.exists( os.path.join('/opt',package)) and not dev:
        raise RuntimeError("%s doesn't appear to be installed in /opt/%s"%(package,package))
    try:
        module = __import__(settings,{},{},settings.split('.'))
    except ImportError as e:
        raise ImportError("%s could not be loaded: %r "%(settings,e))
    else:
        if getattr(module,'DJANGO',True):
            import django
            django.setup()
    return module

class FakeSettings(object):
    """Fake settings that manually sets up very-low-level settings information"""
    def __init__(self):
        product,settings = get_product()
        self.PRODUCT = product
        options = settings.replace('settings','options')
        options = __import__(options,{},{},options.split('.'))
        self.RELEASE = options.getstring('release','revision','SOURCE')
        self.MODIFY_SYSTEM = self.RELEASE != 'SOURCE'

