"""Abstraction point for our encryption routines"""
from __future__ import print_function, unicode_literals
from atxstyle.sixish import unicode, as_unicode
import json, shutil, tempfile, os, logging, sys, subprocess
import glob
from optparse import OptionParser
from fussy import nbio, twrite
from atxstyle import uniquekey, standardlog
from contextlib import contextmanager

log = logging.getLogger(__name__)

ETC_DIR = '/etc/atxlicense'
GLOBAL_KEYS = os.path.join(ETC_DIR, 'keys')
PUBLIC_KEY = 'public.key'
GLOBAL_KEY = 'global-public.key'
GLOBAL_KEY_ABS = os.path.normpath(os.path.join(GLOBAL_KEYS, '..', GLOBAL_KEY))
SIGNING_IDENTITY = 'local-key-identity'
CENTRAL_FLAG = 'is_central'
GPG = '/usr/bin/gpg'
GPG_VERSION = None
NEW_STYLE_GPG = [2, 1]
GPG_STRICT_USER_SPEC = "=%(key)s (Licensing Client) <support@atxnetworks.com>"

ARE_WE_CENTRAL = None
CENTRAL_KEY = 'FC1C2D72AA262F37'


class EncryptionError(ValueError):
    """Error raised on encryption failures"""


class NoEncryptionError(EncryptionError):
    """Error raised when encryption has not been configured"""


@contextmanager
def gpg_keyring_checked(env):
    """
    NOTE(spditner/2019-08-13): Add this 'with' to any calls to the GPG binary
    that will be accessing an existing keyring.

    When invoking GPG commands, it is possible that it will decide to perform
    trustdb maintenance, and can return a non-zero result for that maintenance
    action. For example, you might get an EXIT_CODE of 2 and output on stderr
    such as:

        gpg: checking the trustdb
        gpg: public key of ultimately trusted key C4126891928F025C not found
        gpg: 3 marginal(s) needed, 1 complete(s) needed, PGP trust model
        gpg: depth: 0  valid:   1  signed:   0  trust: 0-, 0q, 0n, 0m, 0f, 1u

    This is a non-fatal operation, but will block other functions such as
    --list-keys from returning non-zero.
    """
    nbio.Process([GPG, '--check-trustdb'], env=env, good_exit=(0, 2))(timeout=60)
    yield


def gpg_version():
    """Get the current GPG version"""
    global GPG_VERSION
    if GPG_VERSION is None:
        GPG_VERSION = [
            int(x, 10)
            for x in nbio.Process(
                [
                    GPG,
                    '--version',
                ]
            )()
            .decode('utf-8')
            .splitlines()[0]
            .split()[-1]
            .split('.')
            if x.isdigit()
        ]
    return GPG_VERSION


def list_keys():
    """Sudo process to list the set of private keys in global keychain"""
    keyring = os.environ.get('GNUPGHOME')
    if not keyring:
        keyring = GLOBAL_KEYS
    try:
        for key in keyring_private_keys(keyring):
            print(key['id'])
    except nbio.ProcessError:
        pass


def parse_private_key_list(content):
    """Given the content of a private-key listing, parse for useful metadata"""
    content = as_unicode(content)
    lines = content.splitlines()
    keys = []
    record = {}  # note: *not* saved...
    for line in lines[2:]:  # two line header...
        if line.startswith('sec '):
            record = {}
            keys.append(record)
            line = line.split()
            record['size'], record['id'] = line[1].split('/')
            record['date'] = line[2]
        elif line.startswith('uid'):
            if not record:
                raise EncryptionError(
                    "Somehow had a start-of-record without a size/date declaration"
                )
            uid = line[3:].strip()
            if uid.startswith('['):
                uid = uid.split(']', 1)[1].strip()
            record['uid'] = uid
    return keys


def keyring_private_keys(keyring):
    env = os.environ.copy()
    env['GNUPGHOME'] = keyring
    # log.debug("Get private key fingerprints for: %s", keyring)
    with gpg_keyring_checked(env):
        return parse_private_key_list(
            nbio.Process(
                [
                    GPG,
                    '--yes',
                    '--fingerprint',
                    '-K',
                    '--keyid-format=long',
                ],
                env=env,
                good_exit=(0, 2),
            )()
        )


def local_identity():
    try:
        base = open(os.path.join(ETC_DIR, SIGNING_IDENTITY)).read().strip()
        if base.startswith('{'):
            return json.loads(base)['id']
        else:
            return id
    except (IOError, OSError):
        log.warning('Unable to read SIGNING_IDENTITY file, prestart not run?')
        return uniquekey.get_base_key()


def keyring_owned(keyring):
    # since it is *euid* that is used to check permissions,
    # we don't want to check *uid* since we may be running
    # in a process that has just temporarily lowered permissions...
    if os.geteuid == 0:
        return True
    uids = [os.geteuid()]
    if not os.stat(keyring).st_uid in uids:
        return False
    for filename in ['secring.gpg', 'pubring.gpg', 'trustdb.gpg']:
        try:
            final = os.path.join(keyring, filename)
            if os.path.exists(final):
                if not os.stat(final).st_uid in uids:
                    return False
        except (IOError, OSError):
            return False
    return True


def are_we_central_licensor(keyring=None, central=None):
    global ARE_WE_CENTRAL
    keyring = keyring or GLOBAL_KEYS
    if ARE_WE_CENTRAL is None:
        if central is None:
            central = CENTRAL_KEY
        if not keyring_owned(keyring):
            try:
                command = 'sudo -n /opt/firmware/current/env/bin/license-list-keys'
                keys = as_unicode(nbio.Process(command)()).splitlines()
            except nbio.ProcessError as err:
                log.exception('Failure running license-list-keys')
                keys = []
        else:
            keys = [key['id'] for key in keyring_private_keys(keyring)]
        ARE_WE_CENTRAL = CENTRAL_KEY in keys
        log.info('Central Licensor? %s %s', ARE_WE_CENTRAL, keys)
    return ARE_WE_CENTRAL


OUR_PUBLIC_KEY = None


def get_our_public_key(keyring=None):
    global OUR_PUBLIC_KEY
    if OUR_PUBLIC_KEY is None:
        public_key = os.path.normpath(
            os.path.join(keyring or GLOBAL_KEYS, '..', PUBLIC_KEY)
        )
        OUR_PUBLIC_KEY = as_unicode(open(public_key).read())
    return OUR_PUBLIC_KEY


def sign_options():
    parser = OptionParser()
    parser.add_option(
        '-k',
        '--key',
        dest='key',
        default=None,
        help="Key identity to use in the signing",
    )
    parser.add_option(
        '-c',
        '--central',
        action='store_true',
        dest='as_central',
        default=False,
    )
    return parser


def sign_main():
    """Sudo-able signature request to sign stdin"""
    standardlog.debug('license-sign', 'firmware')
    parser = sign_options()
    options, args = parser.parse_args()
    content = sys.stdin.read()
    result = sign(content, key=options.key, as_central=options.as_central)
    if hasattr(sys.stdout, 'buffer'):
        # Python 3 writing of binary data
        sys.stdout.buffer.write(result)
    else:
        # Python 2
        sys.stdout.write(result)  # type: ignore
    return 0


def sign(content, key=None, keyring=None, as_central=False):
    """Sign content with our local key"""
    if not isinstance(content, (bytes, unicode)):
        content = json.dumps(content, indent=True)
    if isinstance(content, unicode):
        content = content.encode('utf-8')
    if as_central and key is None:
        key = CENTRAL_KEY
    elif key is None:
        key = local_identity()

    if not key.isalnum():
        raise EncryptionError("Key identities are alpha-numeric only")

    log.info('Signing with: %s', key)

    log.info('Signing %s bytes as %s', len(content), key)
    if keyring is None:
        keyring = GLOBAL_KEYS
    if keyring is GLOBAL_KEYS and not keyring_owned(keyring):
        command = [
            'sudo',
            '-n',
            '/opt/firmware/current/env/bin/license-sign',
            '-k',
            key,
        ]
        if as_central:
            command.append('-c')
        log.info('Spawning sub-process to sign: %s', ' '.join(command))
        return (content | nbio.Process(command))(timeout=60)
    log.info('Keyring is owned, running gpg directly')
    env = os.environ.copy()
    env['GNUPGHOME'] = keyring
    return (content | nbio.Process([GPG, '--yes', '-a', '--sign', '-u', key], env=env))(
        timeout=60
    )


def temp_keyring(public_key):
    """Create a temp keyring with just public_key imported"""
    dir = tempfile.mkdtemp(prefix='sign-check', suffix='gpg')
    env = os.environ.copy()
    env['GNUPGHOME'] = dir
    log.info("Importing public key: %s", public_key)
    (public_key | nbio.Process([GPG, '--yes', '--import'], env=env))()
    return dir


@contextmanager
def keyring_env(key=None, keyring=None):
    if key is None:
        key = open(
            os.path.normpath(os.path.join(keyring or GLOBAL_KEYS, '..', GLOBAL_KEY))
        )
    keyring = temp_keyring(key)
    try:
        env = os.environ.copy()
        env['GNUPGHOME'] = keyring
        yield env
    finally:
        shutil.rmtree(keyring)


def check_signature(content, key=None, keyring=None):
    """Check that signature is by the given public key (not necessarily trusted or imported)"""
    # log.debug("Checking with key %s", key)
    with keyring_env(key, keyring) as env:
        try:
            with gpg_keyring_checked(env):
                keys = as_unicode(nbio.Process([GPG, '--list-keys'], env=env)()).strip()
                for line in keys.splitlines():
                    log.info('KI: %s', line)
                return (content | nbio.Process([GPG, '--yes', '-d'], env=env))()
        except nbio.ProcessError:
            raise EncryptionError('Signature Verification Failed')


def verify_options():
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-k',
        '--key',
        help='File with the GPG public key against which to verify, use global key-dir otherwise',
        default=None,
    )
    return parser


def verify_main():
    """Main function for pipeable gpg verification against root key-server"""
    standardlog.debug('license-verify', 'firmware')
    options = verify_options().parse_args()
    content = sys.stdin.read()
    # TODO: this isn't a good idea, we're letting you use an arbitrary
    # file-path in a sudo operation...
    key = None
    if options.key:
        key = open(options.key).read()
    try:
        result = check_signature(content, key)
    except EncryptionError as err:
        log.error("Decryption failure")
        raise SystemExit(1)
    else:
        if hasattr(sys.stdout, 'buffer'):
            # Python 3 writing of binary data
            sys.stdout.buffer.write(result)
        else:
            # Python 2
            sys.stdout.write(result)  # type: ignore
    return 0


def check_file_signature(filename, key=None, keyring=None, output=None):
    """Check that signature is by the given public key (not necessarily trusted or imported)"""
    with keyring_env(key, keyring) as env:
        with gpg_keyring_checked(env):
            command = [
                GPG,
                '--yes',
                '-d',
            ]
            if output:
                command.extend(['-o', output])
            command.append(filename)
            try:
                return nbio.Process(command, env=env)()
            except nbio.ProcessError:
                raise EncryptionError('Signature Verification Failed')


def decrypt_no_checking(content, **kwargs):
    """Decrypt content speculatively to find declared server key"""
    keyring = tempfile.mkdtemp(prefix='decrypt-unchecked', suffix='gpg')
    try:
        env = os.environ.copy()
        env['GNUPGHOME'] = keyring
        try:
            with gpg_keyring_checked(env):
                return (
                    content
                    | nbio.Process([GPG, '--yes', '-d'], env=env, good_exit=(0, 2))
                )()
        except nbio.ProcessError:
            raise EncryptionError('No valid GPG signature found')
    finally:
        shutil.rmtree(keyring)


def decrypt_from_client(original, **kwargs):
    """Decrypt content from unknown source, first with no checking, then with checking"""
    log.debug("Decrypting to check format and extract claim")
    content = decrypt_no_checking(original, **kwargs)
    log.debug("Extracting claim")
    try:
        structure = json.loads(as_unicode(content))
    except Exception:
        raise ValueError('Certificate could not be parsed')
    server_pk = structure['server_pk']
    if not server_pk:
        raise EncryptionError(
            'Certificate provides null server_pk, available fields: %s', structure
        )
    log.debug("Verifying claim")
    return check_signature(original, server_pk, **kwargs)


# install-time operations


def have_key_file(key_file):
    if not os.path.exists(key_file):
        log.info("Key file missing: %r", key_file)
        return False
    try:
        if os.path.isdir(key_file):
            gpg_exists = [
                filename
                for filename in glob.glob(os.path.join(key_file, '*'))
                if os.stat(filename).st_size
            ]
        else:
            gpg_exists = os.stat(key_file).st_size
            if not gpg_exists:
                log.info("0-byte private key, regenerating")
    except Exception:
        log.exception("Unable to stat the gpg key file")
        gpg_exists = False
    return gpg_exists


def available_identities(key_dir):
    """Get mapping of identifier (email) to key identity"""
    try:
        return keyring_private_keys(key_dir)
    except nbio.ProcessError:
        log.warning("GPG Failure reading keys from %s", key_dir)
        return []
    except (OSError, IOError):
        log.warning("IO Failure reading keys from %s", key_dir)
        return []


def what_identity(key_file):
    """What identity (if any) is our private key defined as using"""
    available = available_identities(key_file)
    if not available:
        raise NoEncryptionError("No encryption somehow")
    # log.debug("Available: %s", available)
    # do we have the private key for our server key?
    for identity in [
        CENTRAL_KEY,
        uniquekey.original_key(),
        uniquekey.baseboard(),
        uniquekey.system(),
        uniquekey.get_base_key(),
    ]:
        if identity:
            identity_full = (
                identity
                if identity == CENTRAL_KEY
                else (GPG_STRICT_USER_SPEC % {'key': identity}).lstrip('=')
            )
            for avail in available:
                if avail['uid'] == identity_full:
                    avail['identity'] = identity
                    return avail
    return None


def create_gpg_cert(key_dir=None, ownership='root:root', keysize=4096):
    ensure_unique_key()
    # anything less that 2.1 is "old style" GPG that used secring
    old_gpg = gpg_version() < NEW_STYLE_GPG
    if key_dir is None:
        key_dir = GLOBAL_KEYS
    log.info("Setting up GPG certificates in %s", key_dir)
    KEYGEN_FILE = os.path.join(key_dir, 'keygen.txt')
    if old_gpg:
        GPG_KEY_FILE = os.path.join(key_dir, 'secring.gpg')
    else:
        GPG_KEY_FILE = os.path.join(key_dir, 'private-keys-v1.d')
    if os.path.islink(GPG_KEY_FILE):
        # sigh...
        log.info("Deleting old key file %s", GPG_KEY_FILE)
        os.remove(GPG_KEY_FILE)

    # log.debug("Checking for keys in %s", GPG_KEY_FILE)
    gpg_exists = have_key_file(GPG_KEY_FILE)
    # log.debug("Have key file: %s", gpg_exists)

    env = os.environ.copy()
    env['GNUPGHOME'] = key_dir
    if not os.path.exists(key_dir):
        log.warning(
            "Key directory does not exist, should have been created in %s", KEYGEN_FILE
        )
        os.makedirs(key_dir)

    identity = current_identity_key = None
    if gpg_exists:
        # private key exists, is one of our possible
        # identities defined as a key within it?
        # Note: there is a bug in later GPG 2.x where it will
        # fail on a hash table update due to a contention
        # if you do a gpg -K then do a generate, so this
        # stanza avoids that for the common case
        try:
            current_identity_key = what_identity(key_dir)
        except NoEncryptionError as err:
            identity = None
            current_identity_key = None
        else:
            if not current_identity_key:
                log.warning(
                    "Our licensing data appears to be for a different mainboard?"
                )
                identity = None
            else:
                identity = current_identity_key
    else:
        # no private key exists, so identity isn't set yet
        current_identity_key = None
        identity = None
    if not identity:
        gpg_exists = False

    if not gpg_exists:
        identity = uniquekey.get_base_key()
        if os.path.exists(KEYGEN_FILE):
            keyfile = open(KEYGEN_FILE).read() % {
                'key': uniquekey.get_base_key(),
                'keysize': keysize,
                'new_comment': '' if old_gpg else '#',
            }
            log.info('Generating GPG Key for identity: %s', identity)
            log.info('KeyGen file:\n%s', keyfile)
            pipe = keyfile | nbio.Process(
                [
                    'gpg',
                    '--batch',
                    '--gen-key' if old_gpg else '--full-gen-key',
                ],
                env=env,
            )
            pipe()
            current_identity_key = what_identity(key_dir)
            if (
                not current_identity_key
                or current_identity_key.get('identity') != identity
            ):
                raise EncryptionError(
                    'We asked to generate an identity for %s, got %s',
                    identity,
                    current_identity_key,
                )
        else:
            log.error("Expected a keygen file in %s", KEYGEN_FILE)
            return

    central_key = os.path.join(key_dir, 'global-public.gpg')
    global_trust = os.path.join(key_dir, 'global-trust.trust')
    if os.path.exists(central_key) and os.path.exists(global_trust):
        if not are_we_central_licensor(key_dir):
            log.info('Importing central licensing server key')
            (open(central_key) | nbio.Process('gpg --import', env=env))()
            (open(global_trust) | nbio.Process('gpg --import-ownertrust', env=env))()
            # nbio.Process(
            #     """gpg --export-ownertrust | sed 's/:.*/:6:/' | gpg --import-ownertrust"""
            # )()
        else:
            log.info('We are central licensor')
            # (
            #     open(global_trust, 'rb')
            #     | nbio.Process('gpg --import-ownertrust', env=env)
            # )()
    if ownership and os.getuid() == 0:
        subprocess.check_call(
            [
                'chown',
                '-R',
                ownership,
                key_dir,
            ]
        )

    subprocess.check_call(
        [
            'chmod',
            '0700',
            key_dir,
        ]
    )
    if os.path.isdir(GPG_KEY_FILE):
        subprocess.check_call(
            [
                'chmod',
                '0700',
                GPG_KEY_FILE,
            ]
        )
    elif os.path.exists(GPG_KEY_FILE):
        subprocess.check_call(
            [
                'chmod',
                '0600',
                GPG_KEY_FILE,
            ]
        )

    # Export keys so that other processes/users can see them...
    license_dir = os.path.join(key_dir.rstrip('/'), '..')
    if current_identity_key:
        key = nbio.Process(
            'gpg -a --export %s' % (current_identity_key['id'],), env=env
        )()
        if key.strip():
            twrite.twrite(os.path.join(license_dir, 'public.key'), key)
        else:
            log.error(
                "Unable to export our key from the private key-store: %s", identity
            )
        twrite.twrite(
            os.path.join(license_dir, 'local-key-identity'),
            json.dumps(current_identity_key, indent=2),
        )
    else:
        raise EncryptionError('Unable to create the signing-key setup')
    if os.path.exists(central_key):
        twrite.twrite(
            os.path.join(license_dir, 'global-public.key'),
            open(central_key).read(),
        )
    return identity


def create_gpg_cert_main():
    standardlog.debug('gpg-setup', do_console=True)
    create_gpg_cert(*sys.argv[1:])


def ensure_unique_key():
    uniquekey.calculate_keys()
