from atxstyle.sixish import unicode, as_unicode, as_bytes
import datetime, subprocess, os, logging, time
from io import StringIO
from functools import wraps
try:
    from urllib import parse
except ImportError:
    from urlparse import urlparse as parse
from fussy import nbio
from lxml import etree

FIRMWARE_HOST = 'https://10.1.0.224/media/firmware/'

log = logging.getLogger(__name__)

__all__ = (
    'ts_name',
    'revno',
    'traced',
    'commit_hash',
    'commit_branch',
    'get_server_links',
    'pull_firmware',
    'get_server_builds',
    'latest_build',
    'fussy_pack_filter',
    'FIRMWARE_HOST',
)


def ts_name(sku, v):
    """Produce a human-readable timestamp (UTC ISO formatted date)"""
    d = datetime.datetime.utcnow()
    if sku == 'ip2a':
        rev = subprocess.check_output(
            'python -c "from ip2a.version import __version__; print(__version__)"',
            shell=True,
        ).strip()
    else:
        rev = v or os.environ.get('BAMBOO_VERSION') or revno()
    branch = commit_branch().strip()
    if branch == 'master':
        branch = ''
    else:
        branch = ''.join([(c if c.isalnum() else '_') for c in branch])
        branch = '-%s' % (branch,)
    return '%s-%s-%s%s' % (d.strftime('%Y-%m-%d'), sku, rev, branch)


def revno():
    """Retrieve the revision number of the main repository"""
    stdout, _ = subprocess.Popen(
        ['git', 'rev-list', '--count', 'HEAD'], stdout=subprocess.PIPE
    ).communicate()
    try:
        return int(stdout)
    except ValueError:
        return 0


def commit_hash():
    try:
        stdout, _ = subprocess.Popen(
            ['git', 'log', '-n', '1'], stdout=subprocess.PIPE
        ).communicate()
        return as_unicode(stdout).splitlines()[0].strip().split()[1]
    except Exception:
        return 'NULL'


def commit_branch():
    try:
        stdout, _ = subprocess.Popen(
            ['git', 'branch'], stdout=subprocess.PIPE
        ).communicate()
        return [
            line.lstrip('*').strip()
            for line in as_unicode(stdout).splitlines()
            if line.startswith('*')
        ][0]
    except Exception:
        return 'unknown'


def get_server_links(product=None):
    """Parse nginx directory listing to find links matching a test function

    yields (link,date,size) for the links in the document that look like directory listings
    """
    import requests, time
    from lxml import etree

    index = FIRMWARE_HOST + (product or env.product) + '/'

    response = requests.get(index, verify=False)
    if not response.ok:
        response.raise_for_status()

    tree = etree.parse(StringIO(as_unicode(response.content)), etree.HTMLParser())
    for link in tree.xpath('//a'):
        target = link.get('href')
        if not (link.tail and link.tail.strip() and link.tail.strip() != '-'):
            continue
        try:
            date, hour, bytes = link.tail.strip().split()
            date = time.mktime(time.strptime('%s %s' % (date, hour), '%d-%b-%Y %H:%M'))
        except Exception as err:
            log.info("Err: %s on %r", err, link.tail)
            continue
        yield parse.urljoin(index, target), date, bytes


def traced(function):
    @wraps(function)
    def with_tracing(*args, **named):
        start_time = time.time()
        log.info("==> %s.%s", function.__module__, function.__name__)
        error = False
        try:
            try:
                return function(*args, **named)
            except Exception as err:
                log.error(
                    "Failure on %s.%s(*%s, **%s) => %s",
                    function.__module__,
                    function.__name__,
                    args,
                    named,
                    err,
                )
                error = err
                raise
        finally:
            log.info(
                "%s %s.%s (%0.1fs)",
                '<ERR' if error else '<==',
                function.__module__,
                function.__name__,
                time.time() - start_time,
            )

    return with_tracing


@traced
def pull_firmware(filename=None, source=FIRMWARE_HOST, target_os=None):
    if filename is None:
        filename = latest_build()
    with_platform = '%s/%s' % (target_os, os.path.basename(filename))
    if filename.endswith(with_platform):
        base = with_platform
    else:
        base = os.path.basename(filename)
    source = source + env.product + '/' + base
    run('mkdir -p /var/firmware/firmware')
    run(
        'wget --no-check-certificate -O /var/firmware/firmware/new %(source)s'
        % locals()
    )


def latest_build(sku=None):
    # lookup on the server
    builds = get_server_builds(product=sku)
    if builds:
        return builds[-1][0]
    raise RuntimeError("Didn't find any builds")
    return sorted(base, key=lambda x: os.stat(x).st_ctime)[-1]


def fussy_pack_filter(target, product):
    """Filter to just report fussy firmwares"""
    return product in target and target.endswith('.gpg') and target.startswith('2')


def get_server_builds(product=None, filter=fussy_pack_filter):
    """Get the set of builds from the .0.224 server"""
    result = []
    for target, date, bytes in get_server_links(product):
        if filter(target, product):
            result.append((target, date, bytes))
    result.sort(key=lambda x: x[1])
    for build in result[-1:]:
        print('Latest build: %s' % (build[0]))
    return result


def run(command, **named):
    log.info('Running: %s', " ".join(command) if isinstance(command, list) else command)
    return nbio.Process(command, **named)()
