from atxstyle.sixishdj import gettext_lazy as _

assert _
import logging
from functools import wraps

from django.contrib import messages
from django.http import HttpResponseRedirect
from django.db import models

log = logging.getLogger(__name__)


def default_session_key(model):
    session_key = model._meta.db_table
    if model._meta.db_tablespace:
        session_key = '%s_%s' % (model._meta.db_tablespace, session_key)
    session_key = '%s_filter_preferences' % (session_key,)
    return session_key


def add_filter_metadata(model, url, session_key=None):
    """Add filter metadata to a give view's requests

    model -- model class being filtered
    url -- name of the url (must be parameter-less) to use for filter updates
        (should eventually call filter_pref below)
    """

    def wrapper(function):
        @wraps(function)
        def with_filter_model_metadata(request, *args, **named):
            request.filtermodel_model = model
            request.filtermodel_url = url
            request.filtermodel_session_key = session_key or default_session_key(model)
            request.filtermodel_current = request.session.get(session_key)
            return function(request, *args, **named)

        return with_filter_model_metadata

    return wrapper


def verify_filter_known(model, field, allowed_traversals):
    """Check if the field name is one of the known fields either through being
    present on the model or allowed in traversals.
    """
    if field:
        if field.lstrip('-') in allowed_traversals:
            return True

        try:
            model._meta.get_field(field.lstrip('-'))
            return True
        except models.FieldDoesNotExist:
            pass

    return False


def filter_pref(
    request,
    model,
    default_next='events',
    session_key=None,
    allowed_traversals=(),
    default_sort=(),
):
    """View to update the user's filter preferences for a given model

    model -- the model class to be filtered
    default_next -- default next view to which to be redirected
        overridden by "next" field in request.POST/GET
    session_key -- string session key, if not provided generated with
        default_session_key
    allowed_traversals -- set of relation traversals allowed in the
        filtering (only relationship traversals *explicitly* specified
        as allowed can be used as filters (to prevent security issues))
    """
    from django.urls import reverse

    REQUEST = request.POST if request.method == 'POST' else request.GET
    next = REQUEST.get('next', reverse(default_next))
    if session_key is None:
        session_key = default_session_key(model)
    if 'reset' in REQUEST:
        request.session[session_key] = {
            'sort': list(default_sort),
        }
    event_prefs = request.session.setdefault(
        session_key,
        {
            'sort': list(default_sort),
        },
    )
    filter_name_prefs = request.session.setdefault('field_names', {})
    # add a filter/exclude
    field = REQUEST.get('field')
    field_label = REQUEST.get('field_label', field)

    if not verify_filter_known(model, field, allowed_traversals):
        messages.error(request, _('We can only sort on known fields'))
        return HttpResponseRedirect(next)

    value = REQUEST.get('value', '').strip()
    for filt in (
        'exclude',
        'filter',
        'after',
        'before',
        'contains',
        'does_not_contain',
    ):
        if filt in REQUEST:
            filters = event_prefs.setdefault(filt, {})
            if filt in ['does_not_contain', 'contains'] and not value:
                try:
                    del filters[field]
                    del filter_name_prefs[field]
                except KeyError:
                    pass
                messages.success(request, 'Removed Filter on %s' % (field_label,))
            else:
                # Ensure we don't duplicate filters
                if field not in filters:
                    filters.setdefault(field, []).append(value)
                    filter_name_prefs.setdefault(field, field_label)
                    messages.success(request, 'Added Filter on %s' % (field_label,))
                else:
                    messages.warning(
                        request, 'Cannot add duplicate filter for: %s' % (field_label,)
                    )
            request.session.save()
            return HttpResponseRedirect(next)
        elif filt + '_remove' in REQUEST:
            filters = event_prefs.setdefault(filt, {})
            current = filters.get(field, [])
            field_label = filter_name_prefs.get(field, field)
            while value in current:
                current.remove(value)
            if not current:
                try:
                    del filters[field]
                    del filter_name_prefs[field]
                except KeyError:
                    pass
            request.session.save()
            messages.success(request, 'Removed Filter on %s' % (field_label,))
            return HttpResponseRedirect(next)
    if 'sort' in REQUEST:
        sort = event_prefs['sort']
        base = field.lstrip('-')
        for old in [base, '-' + base]:
            while old in sort:
                sort.remove(old)
        sort.insert(0, field)
        request.session.save()
        messages.success(request, 'Sorting on %s' % (base,))
        return HttpResponseRedirect(next)
    if 'reset' not in REQUEST:
        # was not a simple reset request, so something we didn't recognize...
        messages.error(request, 'Unrecognized request')
    return HttpResponseRedirect(next)


def base_query(filt, field, value):
    from django.db.models import Q

    if filt == 'filter':
        return Q(**{field: value})
    elif filt == 'exclude':
        return ~Q(**{field: value})
    elif filt == 'after':
        return Q(**{field + "__gte": value})
    elif filt == 'before':
        return Q(**{field + "__lte": value})
    elif filt == 'contains':
        return Q(**{field + "__icontains": value})
    elif filt == 'does_not_contain':
        return ~Q(**{field + "__icontains": value})
    else:
        raise ValueError("Unrecognized filter type %r", filt)


def filter_query(
    query,
    request,
    model,
    session_key=None,
    default_sort=('id',),
):
    """Apply user's configured filtering to the set of query to display/delete

    query -- query to be filtered,

    """
    if session_key is None:
        session_key = default_session_key(model)

    event_prefs = request.session.setdefault(
        session_key,
        {
            'sort': list(default_sort),
        },
    )
    filter_name_prefs = request.session.setdefault('field_names', {})

    q = None
    for filt, properties in event_prefs.items():
        if filt == 'sort':
            continue
        for field, values in properties.items():
            for value in values:
                if q is None:
                    q = base_query(filt, field, value)
                else:
                    q &= base_query(filt, field, value)
    if q:
        try:
            query = query.filter(q)
        except Exception:
            log.exception('Failure creating query')

    sort = event_prefs.get('sort', list(default_sort))
    try:
        query = query.order_by(*sort)
    except Exception as err:
        log.warning("Unable to apply sort:%s", err)
    request.filtermodel_filter = event_prefs
    request.filtermodel_field_names = filter_name_prefs
    return query


from atxstyle.sixishdj import gettext as _

assert _
