"""What is the current bzr revision of a given directory?"""
from __future__ import print_function
import os, sys, glob
from fussy import nbio

def has_changes( directory ):
    directory = os.path.abspath(directory)
    try:
        pipe = nbio.Process( ['bzr', 'diff', directory],cwd=directory )
        if pipe():
            return True 
    except nbio.ProcessError:
        return True
    return False 

def get_root( directory ):
    directory = os.path.abspath(directory)
    pipe = nbio.Process( ['bzr','info', directory],cwd=directory )| nbio.Process( ['grep','branch root:'] )
    result = pipe().splitlines()[0]
    return (result.split(':')[1]).strip()
    
def last_revision( directory ):
    directory = os.path.abspath(directory)
    pipe = nbio.Process( ['bzr','log','-l','1', directory],cwd=directory ) | nbio.Process( ['grep',"^revno:"] )
    result = pipe().splitlines()[0]
    # lines look like:
    # 'revno: 3590 [merge]' 
    # with the merge part optional...
    return int(result.strip().split(':')[1].strip().split()[0])

def revision( directory ):
    if is_bzr( directory ):
        if has_changes( directory ):
            repo_rev = last_revision( get_root( directory ))
            return '%s'%(repo_rev+1)
        else:
            return '%s'%(last_revision( directory ),)
    else:
        root = git_get_root( directory )
        last = git_last_revision(directory)
        if git_has_changes( root ):
            return str( last + 1 )
        return str( last )

def _get_root( directory, marker ):
    original = directory 
    directory = os.path.abspath( directory )
    while directory:
        if os.path.exists( os.path.join( directory, marker )):
            return directory 
        new = os.path.dirname( directory )
        if new == directory:
            break
        directory = new
    raise RuntimeError( original )
    

def bzr_get_root( directory ):
    return _get_root( directory, '.bzr' )
def git_get_root( directory ):
    return _get_root( directory, '.git' )
def git_has_changes( directory ):
    directory = os.path.abspath(directory)
    try:
        nbio.Process( ['git','diff','--quiet', directory],cwd=directory)().strip()
    except nbio.ProcessError:
        return True
    else:
        return False
def git_last_revision( directory = None ):
    directory = os.path.abspath(directory)
    revision = int(nbio.Process( 
        ['git','rev-list','--count','HEAD','--', directory],
        cwd=directory,
    )().strip())
    return revision
def get_last_revision_main( directory = None ):
    if directory is None:
        if sys.argv[1:]:
            directory = sys.argv[1]
        else:
            directory = git_get_root( '.' )
    if is_bzr( directory ):
        print(last_revision( directory ))
    else:
        print(git_last_revision( directory ))
    return 0

def is_bzr( directory ):
    try:
        return bzr_get_root(directory)
    except RuntimeError:
        return False

def main():
    directory = sys.argv[1]
    bzr_rev = revision( directory )
    init_files = glob.glob( os.path.join(directory,'*','__init__.py') )
    for init in init_files:
        version_file = os.path.join( os.path.dirname( init ), 'version.py' )
        if os.path.exists( version_file ):
            lines = open(version_file).read().splitlines()
        else:
            lines = []
        result = []
        written = False
        for line in lines:
            if line.startswith( '#NO-AUTOVERSION'):
                return 
            if line.startswith( '__version__' ):
                version = (line.split('=')[1]).strip().strip('"').strip("'")
                version = version.split('.')[:2]
                version.append( bzr_rev )
                new_revision = '.'.join(version)
                line = '__version__ = %r'%( new_revision )
                written = True
            result.append( line )
        if not written:
            result.append( '__version__ = %r'%( '1.0.%s'%( bzr_rev )) )
        open( version_file, 'w' ).write( '\n'.join( result ))
    
