#! /bin/env python
"""Script to check overall health of the system and log events for bad situations"""
from atxstyle import defaultsettings
defaultsettings.setup_default()
import logging
from optparse import OptionParser
from fussy import cronlock
import psutil

from snmpagents import traps
from atxstyle import standardlog
from atxstyle import system
try:
    from sysemail import clapi
except ImportError as err:
    clapi = None
log = logging.getLogger( __name__ )

DISK_WARN_PERCENTAGE = 90
DISK_ERROR_PERCENTAGE = 98

def check_disks():
    status = system.disk_status()
    for mount in status.get('mounts',[]):
        percent = float(mount.get( 'percent', '0%' ).rstrip('%'))
        if percent > DISK_ERROR_PERCENTAGE:
            log.error( 'Partition %s is full (%s), please remove unused content', mount['mount'],mount['percent'] )
        elif percent > DISK_WARN_PERCENTAGE:
            log.warning( 'Partition %s is close to full (%s), please remove unused content', mount['mount'],mount['percent'] )
    for device in status.get( 'devices', [] ):
        if device.get( 'error', False ):
            log.error( 'Disk Failure (SMART) on %s: %s', device['name'], device['overall'])
            for message in device.get('messages',[]):
                log.error( 'SMART Error Log on %s: %s', device['name'], message )

MEMORY_WARN_PERCENTAGE = 75
MEMORY_ERROR_PERCENTAGE = 90
def check_os():
    status = system.os_status()
    for sensor in status['sensors']['sensors']:
        if sensor.get( 'error' ):
            traps.send( '.1.3.6.1.4.1.16668.50.2.0.1',[
                (
                    '.1.3.6.1.4.1.16668.50.2.2.1.2',
                    'string',
                    sensor['name'],
                ),
                (
                    '.1.3.6.1.4.1.16668.50.2.2.1.3',
                    'string',
                    sensor['current'],
                ),
            ], mibs=None)
            log.error('Temperature sensor %s reports critical temperature (%s)', sensor['name'], sensor['current'] )
            if clapi:
                clapi.send( 
                    'Temperature Error',
                    body='Temperature sensor %s reports critical temperature (%s)'%( sensor['name'], sensor['current']),
                )
                
        elif sensor.get( 'warning' ):
            log.warning( 'Temperature sensor %s reports high temperature (%s)', sensor['name'], sensor['current'] )
    memory = psutil.virtual_memory()
    percent = memory.percent
    if percent > MEMORY_ERROR_PERCENTAGE:
        log.error( "Memory use is extremely high (%s%%)", percent )
        traps.send( '.1.3.6.1.4.1.16668.50.3.0.1',[
            (
                '.1.3.6.1.4.1.16668.50.3.13',
                'integer',
                status['memory'].get('free', 0),
            ),
        ], mibs=None)
        if clapi:
            clapi.send( 
                'Memory Warning',
                body="Memory use is extremely high (%s%%)"%(percent,),
            )
    elif percent > MEMORY_WARN_PERCENTAGE:
        log.warning( "Memory use is high (%s%%)", percent )
    
    if status['load'].get('message'):
        log.warning( status['load']['message'] )
        l1, l5, l15 = status['load']['load']
        # TODO: need a trap for load-level errors
        if status['load'].get('error'):
            if clapi:
                clapi.send( 
                    'Load Warning',
                    body=status['load'].get('message','System load is very high'),
                )
            traps.send( '.1.3.6.1.4.1.16668.50.3.0.2',[
                (
                    '.1.3.6.1.4.1.16668.50.3.15',
                    'integer',
                    int(l5*10),
                ),
                (
                    '.1.3.6.1.4.1.16668.50.3.16',
                    'integer',
                    int(l15*10),
                ),
            ], mibs=None)

def check():
    """Perform all health checks"""
    check_disks()
    check_os()

def health_options():
    parser = OptionParser()
#    parser.add_option( 
#        '-e','--epg', dest='epg_id',
#        help="EPG ID which is being processed" 
#    )
#    parser.add_option( 
#        '-n','--network_ids', dest='network_ids', 
#        help="Comma-separated sequence of network ids to retrieve",
#    )
    return parser 
@cronlock.with_lock( 'health-check', timeout=(3*60))
def main():
    parser = health_options()
    options,args = parser.parse_args()
    standardlog.enable(do_console=False)
    check( )
    return 0
