"""Wrapped json decorators to do common data-type conversions"""
import json, datetime, decimal
from atxstyle import utctime

try:
    # Needed for django 3.x support
    from django.forms.models import ModelChoiceIteratorValue
    from django.db.models import query_utils
except ImportError:
    # Stub that lets us check for isinstance
    query_utils = None

    class ModelChoiceIteratorValue(object):
        pass


# def encode_iterator(i):
#     return [tuple(x) for x in i]
def encode_iterator_value(i):
    return i.value


def encode_memoryview(m):
    return m.tobytes().decode('latin-1')


def encode_deferred(attribute):
    return getattr(attribute.field, 'default', None)


def encode_set(s):
    return sorted(s)


def encode_decimal(d):
    """Closest thing in json"""
    return d.__float__()


def encode_timedelta(d):
    return d.total_seconds()


TYPE_MAP = {
    datetime.datetime: utctime.as_timestamp,
    # ModelChoiceIterator: encode_iterator,
    ModelChoiceIteratorValue: encode_iterator_value,
    set: encode_set,
    datetime.timedelta: encode_timedelta,
    decimal.Decimal: encode_decimal,
}
if query_utils is not None:
    TYPE_MAP[query_utils.DeferredAttribute] = encode_deferred
try:
    memoryview
except NameError as err:
    pass
else:
    TYPE_MAP[memoryview] = encode_memoryview


def encode_pb2(value):
    """Use jsonconversions to encode the given value"""
    from atxnode import jsonconversions

    return jsonconversions.as_json(value)


METHOD_LOOKUP_ORDER = [
    # marker for method to use, method, if none, then use marker attribute
    ('__json__', None),
    ('schedule_json', None),
    ('DESCRIPTOR', encode_pb2),
]


def encoder_for_models(value):
    if value.__class__ in TYPE_MAP:
        return TYPE_MAP[value.__class__](value)
    for marker, handler in METHOD_LOOKUP_ORDER:
        method = getattr(value, marker, None)
        if method:
            try:
                if handler is None:
                    handler = method
                    return handler()
                return handler(value)
            except Exception as err:
                err.args += (handler, value)
                raise
    raise TypeError("Unsupported type for JSON %s" % (value.__class__))


def dumps(value, **named):
    """Dump value to string with common data-types converted

    classes in type-map are used by preference
    then methods in METHOD_LOOKUP_ORDER

    raises TypeError if we don't have an expected record...
    """
    named['default'] = encoder_for_models
    return json.dumps(value, **named)
