"""Generic license-client implementation to reduce code duplication"""
from atxstyle import defaultsettings
defaultsettings.setup_default()
import logging, json, os, shutil
from django.conf import settings
from django.db import transaction
from fussy import twrite, nbio, cronlock
from atxstyle import licenseclient, models, standardlog
log = logging.getLogger(__name__)

def get_metadata():
    """Retrieve metadata for the current product

    Delegates to::

        <product>.config.models.System.get_licensing_metadata()

    which should return a dictionary of extra items
    to add to the base values created in
    :py:mod:`atxstyle.licenseclient`
    """
    system = models.System.get_current()
    metadata = {}
    if hasattr( system, 'get_licensing_metadata' ):
        metadata = system.get_licensing_metadata()
    else:
        log.warning(
            "No get_licensing_metadata() defined on %s.%s",
            system.__class__.__module__,
            system.__class__.__name__,
        )
        if hasattr( system, 'schedule_json' ):
            metadata = system.schedule_json()
    metadata['enable_licensing'] = getattr(system,'ENABLE_LICENSING',True)
    return metadata

def main_options():
    import argparse
    parser = argparse.ArgumentParser(
        description='Downloads licenses from an upstream data-server or imports from local file',
    )
    parser.add_argument(
        '-f','--file',
        help = 'Instead of downloading the certificate from a server, use an existing file',
        default = None,
    )
    return parser

@standardlog.with_debug( 'license-client' )
@cronlock.with_lock( 'license-client', timeout=(20*60))
def main():
    options = main_options().parse_args()
    from atxstyle import encryption
    if encryption.are_we_central_licensor():
        log.info("We are the central server, skipping pull")
        return 0
    try:
        with transaction.atomic():
            metadata_file = licenseclient.LICENSE_SERVER_METADATA
            log.info( 'Collecting metadata and writing to config file' )
            metadata = get_metadata()
            twrite.twrite( metadata_file, json.dumps( metadata ))
        # prevent hanging the DB while we download...
        log.info( 'Running license download client' )
        if settings.RELEASE == 'SOURCE':
            bin_dir = settings.BIN_DIRECTORY
            sudo = []
        else:
            bin_dir = '/opt/firmware/current/env/bin/'
            sudo = ['sudo','-n']
        error = 0
        if options.file:
            shutil.move( options.file, licenseclient.MANUAL_FILE )
            nbio.Process( sudo + [
                os.path.join( bin_dir, 'license-load')
            ])()
        else:
            if metadata.get('enable_licensing'):
                err_log = log.error
            else:
                err_log = log.info
            try:
                nbio.Process( sudo + [
                    os.path.join( bin_dir, 'license-pull')
                ])()
            except nbio.Timeout as err:
                err_log(
                    'Unable to pull licensing metadata from licensing server (timeout)',
                )
                error = 1
            except nbio.ProcessError:
                err_log(
                    'Unable to pull licensing metadata from licensing server',
                )
                error = 1
        with transaction.atomic():
            enforce()
        log.info('Clearing the licensing display cache')
        models.bust_licensing_cache()
        return error
    except Exception:
        log.exception( 'Failure during pull licenses' )
        return 1

def enforce():
    """If System has an enforce_licensing() then call it"""
    system = models.System.get_current()
    if hasattr( system, 'enforce_licensing' ):
        log.info('Starting licensing enforcement')
        system.enforce_licensing()
        log.info('Finished licensing enforcement')
    else:
        log.warning(
            "No enforce_licensing() defined on %s.%s",
            system.__class__.__module__,
            system.__class__.__name__,
        )
