"""Provide (fairly flexible) license model 

Actors:

    * Central Licensor (ATX) VM at ATX (possibly exposed to the world)
    * Licensing Server (CPE) EPGData server for Digistream
    * Dependent Servers (CPE) Digistream servers for Digistream

Each actor has:

    * Central Licensor Public Key (PK)

Use cases:

    * Issue perpetual licenses to a Licensing Server
        * Set expiry date to None 
        * Set a max-standalone period!
            * Otherwise the machines also get a perpetual license and there 
              is no other control available
    * Add new licenses to an existing Licensing Server 
        * Set new levels on the request, send result to client
    * Issue term-limited licenses 
        * Set expiry date to expiry period + grace period 
    * Transfer licenses to a new machine (or reduce/cancel licenses)
        * Issue a NULL license-set for the current machine 
        * User installs null license-set on the current machine 
        * User downloads the new request and uploads to operator
        * Operator verifies that the NULL license is installed 
        * Operator enrolls new machine with the allocations from the old machine

Flow of Licenses:

    * Central Licensor Issues N licenses to Licensing Server
        * Licensing server generates request 
            * Includes all issued certificates
                * [["CERT ID","Client KEY", "ISO-DATE"],...]
            * Is signed by the Licensing Server
            * Is uploaded (binary) to the Central Licensor
            * Is date/time stamped
            * Includes PK
            * Includes Licensing Server ID
            * Includes currently installed certificate (if any)
            * Server will send emails to support (or sales) when licenses are 
              close to expiry
        * Central Licensor validates bundle and issues new certificate 
            * Certificate describes new licensing level 
                * Chosen by the Licensor Operator
                    * Expiry date (optional)
                    * Max standalone period (optional)
                    * Number of Licenses of each type (required)
                * If values are the same (expiry, max standalone, keys, etc)
                    * Include the original certificates 
                    * Otherwise generate new certificates
            * User downloads certificate
                * The Licensing Server can download the certificate itself 
                  if it has a connection and the operator has already 
                  configured its licenses
                    * i.e. two options in the GUI "Download Email Request" or 
                      "Contact Licensing Server Directly"
            * Certificate includes N signed sub-certificates including 
                * License type 
                * Expiry date 
                * Max standalone duration (max time to respect signing by Licensing Server)
                    * i.e. if digistream doesn't contact after N days, the license 
                      is invalidated
                * Licensing Server PK 
                * Licensing Server Key (Hashed MAC)
        * User uploads certificate to Licensing Server 
        * Licensing Server receives Certificate
            * Can issue a "receipt" for current license-set
            * Just a request, technically
        * Server issues individual certificates to machines as assigned
            * Signs attestation including 
                * Issue date 
                * License set 
                * Server ID
            * Server tracks issuance dates and client keys 
                * Simple set of records in the DB
            * Server will warn user about upcoming invalidations
        * Client receives keys
            * Client validates that the individual licenses are valid (Central Licensor PK)
                * Client imports the PK for the Licensing Server encoded in the certificate 
            * Client validates that the license allocation has been signed by the licensing server 
            * Client stores license until invalidated or replaced (next request)
                * Client evaluates duration since issuance periodically and invalidates 
                  certificates which are > max standalone period
            * Client will have N licenses, possibly for different purposes, each is separately 
              validated and verified...
                * EPG 
                * Stream 
                * Encoding/Decoding license files (MPEG-LA licenses)
                    * These will need to be unpacked into the appropriate spaces for Versative
            * Server will warn user about upcoming invalidations

Notes:

    * We assume that all machines are *not* being violated, this is obviously 
      a rather crazy assumption in the face of an unscrupulous attacker, we are 
      concerned with keeping honest customers honest more than preventing outright 
      fraud
"""
from atxstyle.sixishdj import gettext_lazy as _

from django.db import models
from atxstyle import uniquekey
from fussy import nbio
import uuid, json, os


def new_uuid():
    return str(uuid.uuid4())


class License(models.Model):
    class Meta:
        abstract = True

    key = models.CharField(
        max_length=36,
        verbose_name=_("License ID"),
        default=new_uuid,
        unique=True,
    )
    data_server_key = models.TextField(
        verbose_name=_("License/Data Server"),
        help_text=_("The licensing/data-server to which this license is bound"),
        max_length=64,
        default=uniquekey.get_base_key,
    )
    license_json = models.TextField(
        verbose_name=_("Machine Coded License"),
        null=True,
        blank=True,
    )
    signed_license = models.TextField(
        verbose_name=_("Final License Text"),
        null=True,
        blank=True,
    )

    def license_structure(self):
        """Base license to be updated by sub-classes"""
        return {
            'key': self.key,
            'type': self.type,
            'data_server_key': self.data_server_key,
        }

    def from_signed(self, decoded):
        """Take decoded/loaded signed image and load it"""
        self.key = decoded['key']
        self.type = decoded['type']
        self.data_server_key = decoded['data_server_key']
        return self

    def request(self):
        """Create a signing request for the data-server"""
        return json.dumps(self.license_structure(), indent=True)

    def sign(self):
        """Sign content of the license on a license server"""
        if not self.license_json:
            self.license_json = self.request()
        env = os.environ.copy()
        env['GNUPGHOME'] = '/home/digistream/.gnupg'
        self.signed_license = (
            self.license_json | nbio.Process(['gpg', '-a', '--clearsign'], env=env)
        )()

    def load(self, signed_license):
        """Load content of the license on a server

        Uses the /etc/fussy/keys key-ring to validate the source key,
        currently this *only* contains the primary signing key.
        """
        env = os.environ.copy()
        env['GNUPGHOME'] = '/etc/fussy/keys'
        self.license_json = (
            self.signed_license
            | nbio.Process(
                [
                    'gpg',
                    '-d',
                ],
                env=env,
            )
        )()
        return self.from_signed(json.loads(self.license_json))


from atxstyle.sixishdj import gettext as _
