"""Creates a (global) options object from all the .conf files in /etc/%(product)s

from . import loadoptions, getstring, getboolean
options = loadoptions.load( 'productname' )

SOME_VALUE = getstring( options, 'section', 'key', 'Default Value' )
"""
import glob, os
from atxstyle.sixish import unicode
import six

six.add_move(
    six.MovedAttribute(
        'SafeConfigParser',
        'ConfigParser',
        'configparser',
        'SafeConfigParser',
        'ConfigParser',
    )
)
from six.moves import SafeConfigParser
from functools import wraps

THEME_DIRECTORY = '/var/firmware/media/product-theme'
THEME_LINK = os.path.join(THEME_DIRECTORY, 'current')
KEY = 'DJANGO_SETTINGS_MODULE'

DOCUMENTATION = {}


def opt_type(typ):
    def with_opt_type(function):
        function.opt_type = typ
        return function

    return with_opt_type


def save_docstring(function):
    @wraps(function)
    def with_docsave(section, key, default=None, docstring=None):
        result = function(section, key, default, docstring)
        DOCUMENTATION.setdefault(section, {})[key] = {
            'section': section,
            'key': key,
            'default': default,
            '__doc__': docstring,
            'current': result,
            'type': function.opt_type.__name__,
        }
        return result

    return with_docsave


options = None


def loadoptions(product=None):
    global options
    if options is None:
        if product:
            files = '/etc/%(product)s/*.conf' % locals()
        elif KEY in os.environ:
            settings = os.environ.get(KEY)
            product, _ = settings.split('.')
            if product == 'digiconfig':
                product = 'digistream'
            files = '/etc/%(product)s/*.conf' % locals()
        else:
            files = '/etc/firmware/*.conf'
        files = sorted(glob.glob(files))
        if os.path.exists(THEME_LINK):
            files += sorted(glob.glob(os.path.join(THEME_LINK, 'conf', '*.conf')))
        options = SafeConfigParser()
        if files:
            options.read(files)
    return options


@save_docstring
@opt_type(str)
def getstring(section, key, default=None, docstring=None):
    if not options:
        raise RuntimeError("Need to load options first")
    try:
        return options.get(section, key)
    except (
        six.moves.configparser.NoSectionError,
        six.moves.configparser.NoOptionError,
    ):
        return default


@save_docstring
@opt_type(bool)
def getboolean(section, key, default=None, docstring=None):
    if not options:
        raise RuntimeError("Need to load options first")
    try:
        return options.getboolean(section, key)
    except (
        six.moves.configparser.NoSectionError,
        six.moves.configparser.NoOptionError,
    ):
        return default


_NULL = object()


@save_docstring
@opt_type(float)
def getfloat(section, key, default=None, docstring=None):
    value = getstring(section, key, _NULL)
    if value is _NULL:
        return default
    try:
        value = float(value)
    except Exception as err:
        err.args += (section, key, value)
        raise
    else:
        return value


@save_docstring
@opt_type(float)
def getinteger(section, key, default=None, docstring=None):
    value = getstring(section, key, _NULL)
    if value is _NULL:
        return default
    try:
        if isinstance(value, (bytes, unicode)):
            value = int(value, 10)
    except Exception as err:
        err.args += (section, key, value)
        raise
    else:
        return value
