import os, logging, glob, json
from fussy import twrite
from django.conf import settings
from django.http import HttpResponse
from django.shortcuts import render
from functools import wraps
log = logging.getLogger(__name__)
LOCKOUT_DIR = getattr(settings, 'LOCKOUT_DIR', '/var/firmware/run/lockouts')

def lockout_set(lockout, message):
    lockout_file = os.path.join(LOCKOUT_DIR, '.'.join([lockout,'json']))
    twrite.twrite(
        lockout_file,
        json.dumps(dict(error=True, messages=[message,]), indent=2)
    )

def lockout_unset(lockout):
    lockout_file = os.path.join(LOCKOUT_DIR, '.'.join([lockout,'json']))
    try:
        os.unlink(lockout_file)
    except OSError:
        pass

def locked_out(directory=None):
    if not directory:
        directory = LOCKOUT_DIR
    if os.path.isdir(directory):
        return bool(glob.glob(os.path.join(directory,"*.json")))
    return False

def lockout_reasons(directory=None):
    """Produce the set of lockouts currently active"""
    if not directory:
        directory = LOCKOUT_DIR
    messages = list()
    individual = {}
    if os.path.isdir(directory):
        for filename in glob.glob(os.path.join(directory, "*.json")):
            with open(filename) as fh:
                reasons = json.load(fh)
                messages.extend(reasons.get('messages', []))
                individual[os.path.basename(filename)[:-5]] = reasons
    return dict(
        error=bool(messages),
        messages=messages,
        individual = individual,
    )

def locked_with(name):
    """Are we locked with the particular name"""
    reasons = lockout_reasons()
    return reasons['individual'].get(name) is not None


def unless_locked_with(name):
    """Decorate a function such that we check that there are no lockouts of the given name"""
    def with_lock_checker(function):
        @wraps(function)
        def with_lock_check(*args,**named):
            if not locked_with(name):
                return function(*args,**named)
            return HttpResponse(
                json.dumps(reasons),
                content_type = 'application/json',
            )
        return with_lock_check
    return with_lock_checker


def check_for_lockouts( function ):
    """
    TODO(spd/2016-04-11):
        Somehow determine if we are handling a JSON response, or a flat HTML
        response. Will I have to render HTML when blocking? Maybe...

    Use the request to determine whether to insert messages into the result
    dictionary, block posts, or to return them via json.

    Maybe this decorator is a little too smart? :-/
    """
    @wraps( function )
    def with_lockouts( request, *args, **named ):
        if locked_out():
            reasons = lockout_reasons()
            log.debug('Lockouts: %r', reasons)
            if( request.META['HTTP_ACCEPT'].find('application/json') != -1 ):
                formatted = json.dumps(reasons)
                return HttpResponse(formatted, content_type='application/json')
            elif request.method == 'POST':
                return render(
                    request,
                    'atxstyle/lockoutreasons.html',
                    reasons,
                )
            else:
                result = function(request, *args, **named)
                if isinstance(result, dict):
                    result['lockout_reasons'] = reasons
                return result

        else:
            log.debug('No Lockouts')
            return function(request, *args, **named)
    return with_lockouts
