"""Middleware common to our various projects"""
import traceback,logging,time
from django.conf import settings as _settings
log = logging.getLogger( 'GUI.failures' )
from django.http import (
    request,
    HttpResponse,
    HttpResponseRedirect,
)
from atxstyle import requestcache
from django.utils import deprecation
assert requestcache
import json


def get_system():
    """Get the current system object from the "config" application"""
    from atxstyle.djmigrate import get_app
    System = get_app('config').System

    sys = System.objects.first()
    if not sys:
        log.warning("No system object defined! Creating one...")
        sys = System.objects.create()
    return sys


@property
def system( self ):
    """Request property returning the system object"""
    current = getattr(self,'_system',None)
    if not current:
        self._system = current = get_system()
    return current
@property
def settings( self ):
    return _settings
@property
def system_timestamp(self):
    return
@property
def DEBUG(self):
    return _settings.DEBUG
request.HttpRequest.system = system
request.HttpRequest.settings = settings
request.HttpRequest.DEBUG = DEBUG


class SetRemoteAddrFromForwardedFor(deprecation.MiddlewareMixin):
    """
    Middleware that sets REMOTE_ADDR based on HTTP_X_FORWARDED_FOR, if the
    latter is set. This is useful if you're sitting behind a reverse proxy that
    causes each request's REMOTE_ADDR to be set to 127.0.0.1.

    Note that this does NOT validate HTTP_X_FORWARDED_FOR. If you're not behind
    a reverse proxy that sets HTTP_X_FORWARDED_FOR automatically, do not use
    this middleware. Anybody can spoof the value of HTTP_X_FORWARDED_FOR, and
    because this sets REMOTE_ADDR based on HTTP_X_FORWARDED_FOR, that means
    anybody can "fake" their IP address. Only use this when you can absolutely
    trust the value of HTTP_X_FORWARDED_FOR.
    """
    def process_request(self, request):
        try:
            real_ip = request.META['HTTP_X_FORWARDED_FOR']
        except KeyError:
            return None
        else:
            # HTTP_X_FORWARDED_FOR can be a comma-separated list of IPs. The
            # client's IP will be the first one.
            real_ip = real_ip.split(",")[0].strip()
            request.META['REMOTE_ADDR'] = real_ip


class ErrorLogMiddleWare( deprecation.MiddlewareMixin ):
    def process_exception(self, request, exception):
        tb_text = traceback.format_exc()
        url = request.build_absolute_uri()
        log.warning(
            "Unhandled exception encountered on %s\n%s", url,tb_text
        )


class SettingsMiddleWare( deprecation.MiddlewareMixin ):
    def process_request( self, request ):
        """Add DEBUG and settings to the request object"""
        request.system_timestamp = time.time()


class SystemMiddleWare( deprecation.MiddlewareMixin ):
    def allowed_without_eula(self, path):
        for prefix in [
            '/license',
            '/licenses',
            '/system_licenses_json',
            '/accounts/',
            '/sys/licenses/',
            '/lockouts_status_json',
            '/form_bridge/EULAForm',
            '/system_status_json',
            '/api/form_bridge/login/',
            '/ptheme',
        ]:
            if path.startswith(prefix) or path.startswith('/api' + prefix):
                return True
        return False
    def process_request( self, request ):
        """Add the (singleton) system record to the request"""
        #request.system = get_system()
        if request.user.is_authenticated and not request.user.is_superuser:

            # TEMP FIX (burnin only -- until T1060 is implemented
            # ---------------------------------------------------
            if hasattr(_settings, 'PRODUCT') and _settings.PRODUCT == 'burnin':
                return
            if request.system.eula_accepted:
                return
            if not self.allowed_without_eula(request.path):
                log.info("Redirection path %s for EULA",request.path)
                if request.is_ajax():
                    return HttpResponse(json.dumps({
                        'error': True,
                        'messages': [
                            'EULA Must be accepted first',
                        ]
                    }))
                return HttpResponseRedirect('/license/')

from django.db import connection
import logging


class QueryCountMiddleware(deprecation.MiddlewareMixin):
    log = logging.getLogger('querycountmiddleware')

    def process_response(self, request, response):
        if response.status_code == 200:
            total_time = 0

            for query in connection.queries:
                query_time = query.get('time')
                if query_time is None:
                    # django-debug-toolbar monkeypatches the connection
                    # cursor wrapper and adds extra information in each
                    # item in connection.queries. The query time is stored
                    # under the key "duration" rather than "time" and is
                    # in milliseconds, not seconds.
                    query_time = query.get('duration', 0) / 1000
                total_time += float(query_time)

            self.log.info(
                '%s %s %s queries in %s seconds',
                request.method, 
                request.path, 
                len(connection.queries), 
                total_time,
            )
        return response
