from __future__ import unicode_literals
from atxstyle.sixish import unicode
import re, datetime, os, logging, contextlib
from atxstyle.sixishdj import gettext_lazy as _

from django.db import models
from django.conf import settings
from django.core import validators
from django.core.files import File

# from django.utils.functional import lazy

# from django.contrib.contenttypes.models import ContentType
# from django.contrib.contenttypes import generic

from fussy import nbio

from atxstyle import timezones
from videodefs import formatconst as fc

# from videodefs import resolutions
from atxstyle.forms import *
from atxstyle.latechoices import LateChoices
from atxstyle import requestcache

log = logging.getLogger(__name__)

HERE = os.path.dirname(__file__)

MULTIPLE_RE = re.compile(r'^(?P<base>.*?)(\W+\(\d+\))?$')
EULA_ACCEPTED = '/var/firmware/run/eula-accepted.txt'

ROUTING_MAP = {
    # target_cls: [ source_cls, source_cls ]
}

WEB_FORMATS = settings.ETC_OPTIONS.getboolean(
    'encoding',
    'web_formats',
    False,
)


def _comma_string_option(key, default):
    return [
        item
        for item in settings.ETC_OPTIONS.getstring('encoding', key, default)
        .strip()
        .split(',')
        if item
    ]


VIDEO_ENCODINGS = _comma_string_option('video_encodings', ','.join([fc.H264, fc.MPEG2]))
AUDIO_ENCODINGS = _comma_string_option(
    'audio_encodings', ','.join([fc.AC3, fc.MPEG2, fc.MP3, fc.AAC, fc.HEAAC, fc.EAC3])
)
PUBLISH_FORMATS = _comma_string_option('publish_formats', ','.join([fc.SPTS, fc.HLS]))
SUPPORTED_FRAMERATES = _comma_string_option('supported_framerates', '24,25,30000/1001')
TEXT_FORMATS = _comma_string_option(
    'text_formats', 'application/cc-data,private/wss,private/teletext,ANY'
)


def intset_options(option_name, default):
    return [
        int(x)
        for x in settings.ETC_OPTIONS.getstring('encoding', option_name, default)
        .strip()
        .split(',')
    ]


VIDEO_SPEED_PRESETS = intset_options(
    'video_speed_presets', '1,2,3,4,5,6'
)  # 7-9 should only be used for VOD transcoding...
H264_PROFILES = intset_options('h264_profiles', '1,2,3,4,5,6')
MPEG2_PROFILES = intset_options('mpeg2_profiles', '0,1,2,3')
MPEG2_LEVELS = intset_options('mpeg2_levels', '0,2')
ALL_MPEG2_PROFILES = [
    (fc.MPEG_PROFILE_HIGH, _("High")),
    (fc.MPEG_PROFILE_MAIN, _("Main")),
    (fc.MPEG2_PROFILE_HIGH_LOW_LATENCY, _("High Low Latency")),
    (fc.MPEG2_PROFILE_MAIN_LOW_LATENCY, _("Main Low Latency")),
]
ALL_MPEG2_LEVELS = [
    (fc.MPEG_LEVEL_HIGH, _("High")),
    (fc.MPEG_LEVEL_MAIN, _("Main")),
]
# TODO: should be dependent on overall "have arcos" support...

ALL_AUDIO_ENCODINGS = [
    (fc.MPEG2, 'MPEG1-LII (MP2)'),
    (fc.MP3, 'MPEG2-LIII (MP3)'),
    (fc.AAC, 'Advanced Audio Coding (AAC)'),
    (fc.HEAAC, 'High Efficiency AAC (HEAAC)'),
    (fc.AC3, 'AC-3'),
    (fc.PASSTHROUGH_AC3, 'AC-3 Pass Through'),
    (fc.EAC3, 'Enhanced AC-3'),
    (fc.PASSTHROUGH_EAC3, 'EAC-3 Pass Through'),
    # ('passthrough','Passthrough'),
]
ALL_VIDEO_ENCODINGS = [
    (fc.MPEG2, 'MPEG2'),
    (fc.H264, 'H.264'),
    (fc.H265, 'H.265'),
    (fc.ANALOG, 'Analog Out'),
]
ALL_PUBLISH_FORMATS = [
    (fc.SPTS, 'MPEG SPTS'),
    (fc.FLASH, 'Flash'),
    (fc.HLS, 'HTTP Live Streaming'),
    (fc.MP4, 'MP4'),
    (fc.MOV, 'Quicktime (MOV)'),
    (fc.TSX, 'Precompiled TS'),
    (fc.ANALOG, 'Analog Output'),
]
SPTS = fc.SPTS
FLASH = fc.FLASH
HLS = fc.HLS
MP4 = fc.MP4
MOV = fc.MOV
TSX = fc.TSX
ANALOG = fc.ANALOG

# set of content-types exposed to the user...
SCHEDULE_CONTENT = []
SCHEDULE_CONTENT_ALL = []


def content_cls(type):
    """Retrieve a content type from a type specifier (cls.CONTENT_TYPE)"""
    for cls in SCHEDULE_CONTENT_ALL:
        if cls.CONTENT_TYPE == type:
            return cls
    raise TypeError("Unknown content type: %s" % (type))


@requestcache.memoize
def content_object(type, id):
    """Retrieve a content object from a type and id specifier"""
    try:
        cls = content_cls(type)
    except TypeError as err:
        if type in ('default', 'waiting', 'disabled'):
            return None
        err.args += (type, id)
        raise
    try:
        return cls.objects.get(id=id)
    except cls.DoesNotExist:
        return None


def filter_options(options, config, flat=False):
    result = []
    for k, v in options:
        if isinstance(v, list):
            filtered = filter_options(v, config)
            if not filtered:
                continue
            if flat:
                result.extend(filtered)
            else:
                result.append((k, filtered))
        elif k in config:
            result.append((k, v))
    return result


def _produce_names(base):
    yield base
    match = MULTIPLE_RE.match(base)
    if match:
        new_base = match.group('base')
    count = 1
    while True:
        yield u'%s (%s)' % (new_base, count)
        count += 1
        if count > 50000:
            raise RuntimeError("Please choose a different base name")


def find_empty(cls, field, base, current_id=None):
    """Find unused key for field starting with base"""
    all = cls.objects
    if current_id:
        all = all.exclude(id=current_id)
    for name in _produce_names(base):
        if not all.filter(**{field: name}).count():
            return name


def _get_dqam_choices():
    if settings.DQAM:
        from dqam import models as dqam_models
        from django.db import utils

        try:
            # installation time...
            dq = dqam_models.DQAM801B.objects.all()[0]
            return iter(dq.slot_choices())
        except (IndexError, utils.DatabaseError):
            return iter([(i, str(i)) for i in range(1, 9)])
    else:
        # IP stream should not be limited by the dqam's slots...
        return iter([(i, str(i)) for i in range(1, 41)])


dqam_choice_gen = LateChoices(_get_dqam_choices)

"""
Inputs:
    CaptureCard
    EPG
    PPT
    Image
    VOD (w/ Conversions)
    Network
        Segmented Sources
            url
        MPEG-TS/PS
            ip, port, program number
        Depayloading
        Demuxing
        Decoding
        Conditional Access

Outputs:

    ContentStream (Channel)
        Encoding
            Payloading (e.g. udp/rtsp)

    Muxer
        ContentStream -> (ip,port,program)
    DQAM
        (ip,port) -> RF
    Encryption
        1 Network Stream -> 1 Network Stream

"""


def validate_customer_id(value):
    if value:
        fragments = value.split('-')
        if not len(fragments) == 2:
            raise validators.ValidationError(
                "Expected a value like XX-XX (2 values separated by a hyphen)"
            )
        first, second = fragments
        if not first or not second:
            raise validators.ValidationError(
                "Expected a value like XX-XX (2 values separated by a hyphen)"
            )
    return value


def validate_order_number(value):
    if value:
        value = value.upper()
        if len(value) != 6:
            raise validators.ValidationError(
                'Need a 6-digit value or a Q+5-digit value'
            )
        if value[:1] == 'Q':
            rest = value[1:]
            if not rest.isdigit():
                raise validators.ValidationError('Expected value like Q00000')
        else:
            if not value.isdigit():
                raise validators.ValidationError('Expected a value like 000000')


VALID_COMMUNITY = re.compile(r'^[-.# @=:_a-zA-Z0-9]{1,31}$')


def validate_v2c_community(value):
    if not value:
        raise validators.ValidationError('Need a non-null value')
    if len(value) > 31:
        raise validators.ValidationError('Maximum of 31 characters allowed by SNMP v2c')
    if not VALID_COMMUNITY.match(value):
        raise ValueError(
            "%r does not meet requirements for an SNMP v2c community string, characters in the set -.#@=:_ spaces, letters and numbers only"
        )
    return value


def msppt_installed():
    """Is Microsoft PowerPoint Viewer installed?"""
    try:
        return nbio.Process('find /home/digistream/.wine -name "PPTVIEW.EXE"')().strip()
    except nbio.ProcessError:
        return None


def generic_schedule_json(self):
    """Generic schedule JSON extraction"""
    result = generic_description_json(self)
    result.update(
        {
            # Older code expects 'type' instead of __type__
            'type': self.__class__.__name__,
        }
    )
    try:
        from django.contrib.contenttypes.fields import GenericForeignKey
    except ImportError:
        from django.contrib.contenttypes.generic import GenericForeignKey
    for field in [field for field in self._meta.get_fields()]:
        if field.many_to_one:  # foreignkey
            pk = getattr(self, '%s_id' % (field.name,), None)
            if pk is not None:
                result[field.name] = pk
                value = getattr(self, field.name, None)
                if value is not None:
                    result['%s_display' % (field.name)] = str(value)

    for field in [
        field
        for field in self._meta.get_fields()
        if not (
            isinstance(field, GenericForeignKey)
            or isinstance(field, models.BinaryField)
            or field.many_to_many
            or field.one_to_many
            or getattr(field, 'is_relation', False)
        )
    ]:
        try:
            value = getattr(self, field.name)
        except Exception:
            pass
        else:
            display = None
            if isinstance(value, datetime.datetime):
                from atxstyle import utctime

                display = utctime.format_local(value, short=True)
                value = utctime.as_timestamp(value)
            elif isinstance(value, File):
                try:
                    url = None
                    if hasattr(value, 'url'):
                        url = getattr(value, 'url', None)
                    else:
                        method = getattr(self, 'get_%s_url' % (field.name), None)
                        if method:
                            url = method(value)
                    value = {
                        '__type__': 'File',
                        '__pk__': value.path,
                        'friendly_name': os.path.basename(value.name)
                        if value.name
                        else 'Empty',
                        'url': url,
                    }
                except (ValueError, TypeError, IOError, OSError):
                    value = None
            result[field.name] = value
            if display:
                result[field.name + '_display'] = display
            else:
                get_display_name = 'get_%s_display' % (field.name)
                if field.choices or hasattr(self, get_display_name):
                    try:
                        result[field.name + '_display'] = unicode(
                            getattr(self, get_display_name)()
                        )
                    except Exception as err:
                        log.warning(
                            "Failed getting display value of %s.%s: %s",
                            self.__class__.__name__,
                            field.name,
                            err,
                        )
                        result[field.name + '_display'] = unicode(value)

    if hasattr(self, 'get_absolute_url'):
        result['__url__'] = self.get_absolute_url()
    return result


def generic_reference_json(self, attributes):
    """Do generic addition of ForeignKey references via attributes"""
    result = {}
    if self.id:
        for attr in attributes:
            value = getattr(self, attr, None)
            if value is None:
                result[attr] = None
                result['%s_display' % (attr,)] = ''
            else:
                if isinstance(value, models.Manager):
                    value = value.all()
                    result[attr] = [v.pk for v in value]
                    result['%s_display' % (attr,)] = ', '.join([str(v) for v in value])
                else:
                    result[attr] = value.pk
                    result['%s_display' % (attr,)] = str(value)
                    if hasattr(value, 'get_absolute_url'):
                        value_url = value.get_absolute_url()
                        result['%s_url' % (attr,)] = value_url
                        # log.info('Calculated url for %s: %s', value, value_url)
    return result


def generic_description_json(self):
    """Just enough information to describe the given component (e.g. to load a form or a view)"""
    base = {
        '__key__': js_key(self),
        'type_name': getattr(self.__class__._meta, 'verbose_name')
        or self.__class__.__name__,
        'title': unicode(self),
        '__type__': self.__class__.__name__,
        '__pk__': self.pk,
        '__version__': getattr(self, 'config_version', 0),
    }
    if hasattr(self, 'get_absolute_url'):
        base['__url__'] = self.get_absolute_url()
    if (
        all(
            [hasattr(self, x) for x in ['tree_id', 'lft', 'rght', 'level', 'parent_id']]
        )
        and self.id
    ):
        base['tree'] = {
            'node_id': self.id,  # this is what parent_id actually references, *not* our tree_id
            'parent_id': self.parent_id,
            # Django mptt internal stuff that's needed for rendering and ordering...
            'level': self.level,
            'lft': self.lft,
            'rght': self.rght,
            'tree_id': self.tree_id,  # this is the "tree namespace", not the id within the tree
        }

    return base


class AbstractContent(object):
    """Mix-in for all content objects"""

    CONTENT_TYPE = 'content'
    FRIENDLY_TYPE_NAME = 'Content'
    CONTENT_CAN_CREATE = False
    CONTENT_CAN_DELETE = False

    @classmethod
    def content_type_json(cls):
        """Produce the content-type's json description"""
        return {
            'type': unicode(cls.CONTENT_TYPE),
            'type_name': unicode(cls.FRIENDLY_TYPE_NAME),
            'can_delete': cls.CONTENT_CAN_DELETE,
            'can_create': cls.CONTENT_CAN_CREATE,
            'extensions': [unicode(x) for x in getattr(cls, 'ACCEPTED_EXTENSIONS', [])],
            'mime_types': [unicode(x) for x in getattr(cls, 'ACCEPTED_TYPES', [])],
            'disallowed_subtypes': [
                unicode(x) for x in getattr(cls, 'DISALLOWED_CONTENT_TYPES', [])
            ],
        }

    @classmethod
    def cache_all(cls):
        """Cache all of our instances for fast lookup"""
        for instance in cls.objects.all():
            requestcache.set(
                (content_object.__original__, (cls.CONTENT_TYPE, instance.id)),
                instance,
            )


def js_key(object):
    if isinstance(object, type):
        return object.__name__
    if hasattr(object, 'CONTENT_TYPE'):
        return '%s.%s' % (object.CONTENT_TYPE, object.id)
    else:
        return '%s.%s' % (object.__class__.__name__, object.id)


class TZField(models.CharField):
    description = "Character field holding timezone selection from OS timezones"

    def deconstruct(self):
        name, path, args, kwargs = super(TZField, self).deconstruct()
        # Ignore choice changes when generating migrations
        kwargs.pop('choices', None)
        return (name, path, args, kwargs)


class System(models.Model):
    have_msppt = staticmethod(msppt_installed)

    class Meta:
        abstract = True

    FACTORY_SERIAL_FORM_FIELDS = (
        'serial_number',
        'customer_id',
        'order_number',
    )
    location = models.CharField(
        max_length=255,
        verbose_name=_("Physical Location"),
        help_text=_(
            "Description of how to find this machine (physically), e.g. Head End, Rack, Slot"
        ),
        default='',
        blank=True,
    )
    timezone = TZField(
        default='US/Eastern',
        choices=timezones.TIMEZONE_CHOICES,
        max_length=timezones.LONGEST_TIMEZONE,
    )
    periodic_reboot = models.IntegerField(
        verbose_name=_("Periodic Reboot"),
        default=0,
        help_text=_("Enables periodic reboot of the machine"),
        choices=[
            (0, _('Disabled')),
            (1, _('Daily')),
            (2, _('Weekly (Sunday)')),
        ],
    )
    periodic_reboot_time = models.CharField(
        default='00:00',
        validators=[
            time_validator,
        ],
        help_text=_(
            "Time-of-day at which periodic reboot should occur in 24-hour HH:MM notation, weekly reboot occurs on Sundays"
        ),
        verbose_name=_("Periodic Reboot Time"),
        max_length=5,
    )

    def periodic_reboot_description(self):
        if not self.periodic_reboot:
            return 'No periodic reboot'
        return _('Reboot %(frequency)s at %(hour)s') % {
            'frequency': self.get_periodic_reboot_display(),
            'hour': self.periodic_reboot_time,
        }

    atx_license_server = models.URLField(
        verbose_name=_("License Server URL"),
        null=True,
        blank=True,
        help_text=_(
            "Server URL (e.g. https://digistreamepgdata.atxnetworks.com) from which to pull ATX License Bundles for this machine"
        ),
        max_length=255,
    )
    eula_accepted = models.DateTimeField(
        verbose_name=_("EULA Accepted"),
        null=True,
        blank=True,
        help_text=_("Date on which EULA was accepted"),
    )
    ATX_LICENSE_URL_PATH = '/certs/client/%s/certificates/v2/'

    @classmethod
    def uniquekey(cls):
        from atxstyle import uniquekey

        return uniquekey.get_base_key()

    @classmethod
    def old_uniquekey(cls):
        from atxstyle import uniquekey

        return uniquekey.original_key()

    def save(self, *args, **named):
        """Override save to write out ATX_LICENSE_CLIENT information if necessary"""
        from atxstyle import licenseclient

        result = super(System, self).save(*args, **named)
        current = licenseclient.current_license_server_url()
        new = self.atx_license_server or None
        if new:
            new = new.rstrip('/') + self.ATX_LICENSE_URL_PATH
        if new and current != new:
            licenseclient.set_license_server_url(new)
        return result

    @property
    def timezone_obj(self):
        import pytz

        return pytz.timezone(self.timezone)

    @property
    def time(self):
        return datetime.datetime.now(tz=self.timezone_obj)

    @property
    def time_formatted(self):
        return self.time.strftime('%Y-%m-%d %H:%M')

    snmp_community = models.CharField(
        verbose_name=_("Read Community"),
        help_text=_("Read-only community for snmp monitoring"),
        default='public',
        validators=[
            validate_v2c_community,
        ],
        max_length=64,
    )
    snmp_rw_community = models.CharField(
        verbose_name=_("Write Community"),
        help_text=_("Read-write community for snmp monitoring"),
        default='blueMoo32C',
        validators=[
            validate_v2c_community,
        ],
        max_length=64,
    )
    snmp_trap_sink = models.GenericIPAddressField(
        verbose_name=_("SNMP Trap Server"),
        help_text=_("Server to which to send traps"),
        default='127.0.0.1',
    )
    snmp_trap_sink_port = models.IntegerField(
        verbose_name=_("SNMP Trap Port"),
        help_text=_("IP Port to which  SNMP v2c traps will be sent"),
        default=162,
        validators=[
            validate_ip_port_range,
        ],
    )
    snmp_trap_community = models.CharField(
        verbose_name=_("SNMP Trap Community"),
        help_text=_("SNMP Community for the trap server"),
        default='public',
        validators=[
            validate_v2c_community,
        ],
        max_length=64,
    )
    serial_number = models.CharField(
        verbose_name=_("Serial Number"),
        help_text=_("ATX System Serial Number for this host/chasis"),
        null=True,
        blank=True,
        max_length=64,
    )
    customer_id = models.CharField(
        verbose_name=_("Customer ID"),
        help_text=_("ATX Customer identifier, such as TW-CH1"),
        null=True,
        blank=True,
        max_length=12,
        validators=[
            validate_customer_id,
        ],
    )
    order_number = models.CharField(
        verbose_name=_("Sales Order"),
        help_text=_("ATX Sales Order Number (6-digits)"),
        null=True,
        blank=True,
        max_length=6,
        validators=[
            validate_order_number,
        ],
    )

    def write_periodic_reboot(self):
        """Run sudo promotion to write out the periodic reboot configuration"""
        promote = os.path.join(settings.BIN_DIRECTORY, 'promote-periodic-boot')
        hour, minute = self.periodic_reboot_time.split(':')
        command = [
            'sudo',
            '-n',
            promote,
            str(self.periodic_reboot),
            str(hour),
            str(minute),
        ]
        if settings.MODIFY_SYSTEM:
            nbio.Process(command)()
        else:
            log.warning("Would run: %s", " ".join(command))
        return True

    def write_snmp_config(self):
        """Writes both SNMP configuration and device-status json file"""
        from . import writedevicestatus

        log.info("Writing system status data-file")
        writedevicestatus.write(system=self)
        if settings.MODIFY_SYSTEM:
            for cmd in ['snmp-writeconf']:
                final = os.path.join(settings.BIN_DIRECTORY, cmd)
                if os.path.exists(final):
                    try:
                        nbio.Process(
                            [
                                'sudo',
                                '-n',
                                final,
                            ]
                        )()
                    except nbio.ProcessError as err:
                        log.error(
                            'Failure running %s',
                            final,
                        )
                else:
                    log.error("Expected config program: %s not present", final)
        else:
            for cmd in ['snmp-writeconf']:
                final = os.path.join(settings.BIN_DIRECTORY, cmd)
                log.info("Would run: sudo -n %s", cmd)

    def sku_json(self):
        return {
            'product': settings.PRODUCT,
            'sku': getattr(settings, 'SKU', settings.PRODUCT),
            'sku_name': getattr(settings, 'SKU_NAME', settings.PRODUCT),
            'release': settings.RELEASE,
        }

    # If licensing is enabled, we consider failures to acquire
    # the license an error to report to the user, otherwise we
    # just log the failures and ignore them
    ENABLE_LICENSING = True

    def schedule_json(self):
        return {
            'key': js_key(self),
            '__type__': self.__class__.__name__,
            '__pk__': self.pk,
            'type': self.__class__.__name__,
            'id': self.id,
            'time_formatted': self.time_formatted,
            'friendly_description': self.location
            or self.serial_number
            or settings.PRODUCT,
            'all_stop': getattr(self, 'all_stop', None),
            'reboot_required': self.reboot_required(),
            'power': {
                'reboot': getattr(settings, 'POWER_REBOOT', True),
                'power_cycle': getattr(settings, 'POWER_POWERCYCLE', True),
                'shutdown': getattr(settings, 'POWER_SHUTDOWN', True),
            },
            'system': {
                'location': self.location,
                'timezone': self.timezone,
                'serial': self.serial_number,
                'order_number': self.order_number,
                'customer_id': self.customer_id,
                'periodic_reboot': self.periodic_reboot,
                'periodic_reboot_display': self.periodic_reboot_description(),
                'periodic_reboot_time': self.periodic_reboot_time,
                'license_server': self.atx_license_server,
                'eula_accepted': self.eula_accepted.strftime('%Y-%m-%d')
                if self.eula_accepted
                else None,
            },
            'sku': self.sku_json(),
            'snmp': {
                'snmp_community': self.snmp_community,
                'snmp_rw_community': self.snmp_rw_community,
                'snmp_trap_community': self.snmp_trap_community,
                'snmp_trap_sink': self.snmp_trap_sink,
                'snmp_trap_sink_port': self.snmp_trap_sink_port,
            },
            'enable_licensing': self.ENABLE_LICENSING,
        }

    def public_schedule_json(self):
        base = self.schedule_json()
        del base['snmp']['snmp_rw_community']
        return base

    __json__ = public_schedule_json

    def from_schedule_json(self, structure):
        """Apply schedule json structure to this system record"""
        for root, properties in [
            (
                'system',
                [
                    ('location', 'location'),
                    ('timezone', 'timezone'),
                    ('serial', 'serial_number'),
                    ('order_number', 'order_number'),
                    ('customer_id', 'customer_id'),
                    ('periodic_reboot', 'periodic_reboot'),
                    ('periodic_reboot_time', 'periodic_reboot_time'),
                    ('license_server', 'license_server'),
                ],
            ),
            (
                'sku',
                [
                    # currently only higher-level code can apply the
                    # SKU settings...
                ],
            ),
            (
                'snmp',
                [
                    ('snmp_community', 'snmp_community'),
                    ('snmp_rw_community', 'snmp_rw_community'),
                    ('snmp_trap_community', 'snmp_trap_community'),
                    ('snmp_trap_sink', 'snmp_trap_sink'),
                    ('snmp_trap_sink_port', 'snmp_trap_sink_port'),
                ],
            ),
        ]:
            record = structure.get(root)
            if record:
                for key, prop in properties:
                    if key in record:
                        setattr(self, prop, record[key])
        self.save()
        return self

    @staticmethod
    @requestcache.memoize
    def get_final():
        """Retrieve the final product-specific system object"""
        from atxstyle.djmigrate import get_app

        return get_app('config').System

    @staticmethod
    @requestcache.memoize
    def get_current():
        cls = System.get_final()
        for record in cls.objects.all():
            return record
        return cls.objects.create()

    @property
    def current_sysemail(self):
        from sysemail import models

        return models.SMTP.current()

    @classmethod
    @requestcache.memoize
    def reboot_required(cls):
        from atxstyle import rebootrequired

        return rebootrequired.get()


def load_iso6392_choices():
    import json

    return json.loads(open(os.path.join(HERE, 'iso6392.json')).read())


def group_schedule_json(group):
    from atxstyle import ownership

    non_ownership = ownership.non_ownership_groups()
    is_ownership = not (group.name in non_ownership)
    permission = group.name in ownership.permission_groups()
    help = ownership.group_help(group.name)
    return {
        'type': 'Group',
        'id': group.id,
        'name': group.name,
        'ownership': is_ownership,
        'permission': permission,
        'help': help,
    }


def get_protected_users():
    protected_users = []
    if 'users' in settings.DEFAULT_PERMISSIONS and isinstance(
        settings.DEFAULT_PERMISSIONS['users'], dict
    ):
        protected_users = settings.DEFAULT_PERMISSIONS['users'].keys()

    return protected_users


def is_user_deletable(user):
    return not user.is_superuser and user.get_username() not in get_protected_users()


def user_schedule_json_base(user):
    return {
        'type': 'User',
        'id': user.pk,
        'is_anonymous': bool(
            user.is_anonymous
        ),  # django 2.x this is no longer a callable
        'title': user.get_username(),
        'is_superuser': bool(user.is_superuser),
        'is_staff': bool(user.is_staff),
        'is_active': bool(user.is_active),
        'name': user.get_username(),
        'username': user.username,
        'first_name': getattr(user, 'first_name', 'Anonymous'),
        'email': getattr(user, 'email', None),
        'last_name': getattr(user, 'last_name', None),
        'last_login': str(user.last_login)
        if getattr(user, 'last_login', None)
        else 'never',
        'editable': getattr(user, 'editable', False),
        'deletable': is_user_deletable(user),
        'group_names': [g.name for g in user.groups.all()] if user.id else [],
    }


def user_schedule_json(user):
    base = user_schedule_json_base(user)
    base.update(
        {
            'groups': [
                group_schedule_json(group)
                for group in user.groups.order_by('name').all()
            ]
            if user.id
            else [],
            'permissions': sorted(user.get_all_permissions()) if user.id else [],
        }
    )
    return base


from django.contrib.auth import models as auth_models


def user_get_absolute_url(user):
    from django.urls import reverse

    return reverse('user', kwargs=dict(user=user.id))


auth_models.User.get_absolute_url = user_get_absolute_url


@contextlib.contextmanager
def rollback(*args, **named):
    from django.db import transaction, IntegrityError

    class Rollback(IntegrityError):
        pass

    try:
        with transaction.atomic(*args, **named):
            yield
            raise Rollback
    except Rollback:
        log.info("Performed rollback")
        pass


def bust_licensing_cache(**named):
    """Bust the cache of licensing metadata"""
    try:
        from django.core.cache import cache
        from django.core.cache.utils import make_template_fragment_key

        key = make_template_fragment_key('fragment_license_summary')
        cache.delete(key)
    except KeyError:
        pass


from atxstyle.sixishdj import gettext as _
