"""Setup node modules *outside* of the source code tree with our current requirements

Uses the package definition in `js/package.json` to control the
installation of *both* production and development packages.

This avoids two issues:

* for IDEs that automatically index and wach every sub-directory,
  avoids having them become crazily slow
* ties the node setup to a particular virtualenv by installing the
  relevant node modules into the virtualenv itself and making them
  part of the path of the virtualenv
"""
import os, logging
from osupdates import ostarget
from fussy import nbio

HERE = os.path.dirname(__file__)
PARENT = os.path.abspath(os.path.join(HERE, '..', '..', '..'))

SOURCE_DIR = os.path.abspath(os.path.join(HERE, '..', '..'))
SOURCE_JS = os.path.join(SOURCE_DIR, 'js')
log = logging.getLogger(__name__)

TARGET = ostarget.local_target()
TARGET.DEFAULT_TIMEOUT = 60 * 10


def current_node_version(target=TARGET):
    try:
        version = target.run('node --version').decode('utf-8').strip()
        log.info("Current NodeJS version: %s", version)
    except nbio.ProcessError:
        version = 'v0'
    version = [int(x) for x in version.strip('v').split('.')]
    return tuple(version)


def get_venv():
    if 'VIRTUAL_ENV' in os.environ:
        venv_dir = os.environ['VIRTUAL_ENV']
    else:
        source_dir_name = os.path.basename(SOURCE_DIR)
        venv_dir_name = source_dir_name + '-env'
        venv_dir = os.path.join(PARENT, venv_dir_name)
    return venv_dir


def get_js_dir():
    return SOURCE_DIR


def get_node_dir():
    return os.path.join(get_js_dir(), 'node_modules')


def add_path_to_venv():
    activate_script = os.path.join(get_venv(), 'bin', 'activate')
    node_home = get_node_dir()
    node_bin = os.path.join(node_home, '.bin')

    if os.path.exists(os.path.join(PARENT, 'node_modules')):
        os.rename(
            os.path.join(PARENT, 'node_modules'),
            os.path.join(PARENT, 'node_modules-junk'),
        )
        log.warning(
            "Old node_modules detected, you can delete %s",
            os.path.join(PARENT, 'node_modules-junk'),
        )

    if os.path.exists(activate_script):
        path_string = '\nPATH="{}:$PATH"; export PATH'.format(node_bin)
        if path_string not in open(activate_script).read():
            cmd = "echo '{}' >> {}".format(path_string, activate_script)
            os.system(cmd)
            log.info(" UPDATED VENV PATH (to include %s) ", node_bin)
        else:
            log.info(" Already have node in venv path ")
    else:
        log.warning(" UNABLE TO UPDATE VENV PATH ")


def node_install(target=TARGET):
    """Download nodejs v 6.7+ and pnpm via package manager"""
    log.info("Checking NodeJS version...")
    version = current_node_version(target)
    if (target.name != 'precise') and version < (8,):
        target.install_node()
    else:
        log.info("NodeJS already installed. Moving on...")

    # Update NPM to the latest version
    try:
        version = [
            int(x)
            for x in nbio.Process('pnpm --version')().decode('utf-8').strip().split('.')
        ][:3]
    except nbio.ProcessError:
        version = [0, 0]
    if version < [7]:
        target.sudo('npm install pnpm@latest -g')
    else:
        log.info("Already have modern pnpm installed")

    # Node binary locations added to the path in the venv...
    add_path_to_venv()
    if target.name == 'centos7':
        target.dependencies(['gcc-c++', 'gcc', 'make', 'cmake'])
    log.info("Doing package install with pnpm")
    target.run('pnpm install --shamefully-hoist ./', cwd=get_js_dir())


def main():
    logging.basicConfig(level=logging.INFO)
    node_install()


if __name__ == "__main__":
    main()
