"""Interface to the Numato USB Powered Relay device (via PySerial)"""
import time, logging, glob, os
from serial import Serial
#from serial import SerialException, SerialTimeoutException
log = logging.getLogger( __name__ )

class SerialIOError( IOError ):
    pass
class NoNumatoError( RuntimeError ):
    pass

class NumatoRelay( object ):
    newline = '\r'
    prompt = '>'
    def __init__(self, port=None, timeout=3.0, writeTimeout=1.0):
        if port is None:
            # Linux specific...
            try:
                port = sorted(glob.glob( '/dev/ttyACM[0-9]' ))[0]
            except IndexError:
                raise NoNumatoError( 'No Numato Relay could be found' )
        elif isinstance( port, (int,long) ):
            port = '/dev/ttyACM%s'%( port, )
            if not os.path.exists( port ):
                raise NoNumatoError( 'Port %s not present'%(port,))
        self.port = Serial(port, 9600, timeout=timeout, writeTimeout=writeTimeout)
        self.write( self.newline )
        self.read_until( self.prompt )
    def close( self ):
        if self.port:
            self.port.close()
            self.port = None
    def write( self, output ):
        if not self.port:
            raise RuntimeError( "Attempt to use a closed NumatoRelay" )
        self.port.flushOutput()
        self.port.flushInput()
        if not output.endswith( self.newline ):
            output += self.newline
        log.debug( 'out: %r', output )
        attempts = 0
        while output and attempts < 200:
            output = output[self.port.write( output ):]
            attempts += 1
            if output:
                time.sleep( .01 * attempts )
        if output:
            raise SerialIOError( "Unable to write command" )
    def read_until( self, delimiter ):
        if not self.port:
            raise RuntimeError( "Attempt to use a closed NumatoRelay" )
        buf = ''
        while True:
            c = self.port.read()
            if c:
                buf += c
                if buf.endswith( delimiter ):
                    buf = buf[:-len(delimiter)]
                    log.debug('in: %r', buf)
                    return buf
            else:
                raise SerialIOError( "Timeout reading from port" )
    
    def command( self, command ):
        self.write( command )
        return self.as_lines( self.read_until( self.newline+self.prompt ) )
    
    def as_lines( self, result ):
        return [
            line for line in [l.strip() for l in result.strip().splitlines()]
            if line 
        ]
    def text_boolean( self, value ):
        if value == 'on':
            return True 
        elif value == 'off':
            return False
    
    # concrete commands below
    def ver( self ):
        """Retrieve firmware version"""
        result = self.command( 'ver' )
        return result[1]
    def relay( self, number=0, command='read' ):
        """General relay command"""
        if command not in ('read','on','off'):
            raise ValueError( 'Unrecognized command' )
        result = self.command( 'relay %(command)s %(number)s'%locals())
        if command == 'read':
            return self.text_boolean(result[1])
        return result
    def on( self, number=0 ):
        """Turn relay <number> on"""
        return self.relay( number, 'on' )
    def off( self, number=0 ):
        """Turn relay <number> off"""
        return self.relay( number, 'off' )
    def read( self, number=0 ):
        """Read current state of relay <number> -> bool"""
        return self.relay( number, 'read' )
    
    def gpio( self, number=0, command='read' ):
        """read/clear/set GPIO pin <number> -> int (for read)"""
        if command not in ('read','clear','set'):
            raise ValueError( 'Unrecognized command' )
        result = self.command( 'gpio %(command)s %(number)s'%locals())
        if command == 'read':
            return int(result[1])
        return result
    
    def adc( self, number=0 ):
        """read analog-to-digital converter <number> -> int"""
        return int(self.command( 'adc read %(number)s'%locals() )[1])
        
def test_numato_relay(port):
    relay = NumatoRelay(  )
    try:
        version = relay.ver()
        log.info( 'Firmware: %r',version )
        log.info( 'Relay 0: %s', relay.read())
        log.info( 'Relay 0 on: %s', relay.on())
        log.info( 'Relay 0: %s', relay.read())
        log.info( 'Relay 0 off: %s', relay.off())
        log.info( 'Relay 0: %s', relay.read())
        for i in range(4):
            current = relay.gpio( i, 'read' )
            log.info( 'GPIO %s: %s', i, current )
            relay.gpio( i, ['clear','set'][not bool(current)] )
            current = relay.gpio( i, 'read' )
            log.info( 'GPIO Flip %s: %s', i, current )
            
            log.info( 'ADC %s: %s', i, relay.adc(i))
    finally:
        relay.close()

def main():
    logging.basicConfig( level=logging.INFO )
    test_numato_relay(None)

if __name__ == "__main__":
    main()
