"""Ownership cross-cutting behaviour"""
import functools
import django.core.exceptions
from django.db.models import Q
from django.contrib.auth import models
from atxstyle.sixish import lru_cache


def constrained_user(user):
    """Is this user a constrained user?

    You are constrained if you are not one of:

    - a superuser
    - staff
    - a user with a permission config.System.own_everything

    returns True/False
    """
    return not (
        user.is_superuser or user.is_staff or user.has_perm('config.own_everything')
    )


def filter_query(query, user, field='owner'):
    """Apply ownership filter to the given query

    field -- you can provide e.g. `group__owner` to filter
             on a related table...

    If you are a constrained user, filter the query's owner
    field to be limited to your group ids.
    """
    if constrained_user(user):
        try:
            query = query.filter(
                **{'%s__in' % (field,): [group.id for group in user.groups.all()]}
            )
        except (django.core.exceptions.FieldError) as err:
            err.args += (query,)
            raise
    return query


def user_owns(record, user, field='owner'):
    """Can this user access this record (ownership check)"""
    if constrained_user(user):
        owner = getattr(record, field, None)
        if not owner:
            # Not owned by anyone
            return False
        return owner in user.groups.all()
    return True


def form_field_filter(form, key, user, filter_field='owner'):
    """Filter the queryset for field key on form"""
    if constrained_user(user):
        queryset = form.fields[key].queryset
        form.fields[key].queryset = filter_query(queryset, user, field=filter_field)
    return form.fields[key].queryset


def form_field_filter_owner(form, user, filter_field='owner'):
    """What groups can this user assign to own a record

    If the user is constrained, only those to which they
    are a member.

    Even if unconstrained, do not allow system groups to
    own if they have not been explicitly marked as being
    allowed to own things.
    """
    if constrained_user(user):
        queryset = form.fields[filter_field].queryset
        base = queryset.filter(id__in=[group.id for group in user.groups.all()])
    else:
        base = form.fields[filter_field].queryset
    form.fields[filter_field].queryset = base.exclude(
        name__in=non_ownership_groups(),
    ).order_by('name')
    return form.fields[filter_field].queryset


def ownership_groups_query(user):
    # TODO: bad definition of "what is a permission group vs an owner group" here

    perm_groups = non_ownership_groups()
    our_groups = user.groups.exclude(
        name__in=perm_groups,
    )
    return our_groups


@lru_cache(maxsize=1)
def non_ownership_groups():
    from django.conf import settings

    return set(
        [
            key
            for key, value in settings.DEFAULT_PERMISSIONS['groups'].items()
            if not value.get('can_own')
        ]
    )


@lru_cache(maxsize=1)
def permission_groups():
    from django.conf import settings

    return set(
        [
            key
            for key, value in settings.DEFAULT_PERMISSIONS['groups'].items()
            if value.get('permissions')
        ]
    )


@lru_cache(maxsize=256)
def group_help(group_name):
    """See if there is a group help description"""
    from django.conf import settings

    group = settings.DEFAULT_PERMISSIONS['groups'].get(group_name, {})
    if not group:
        return 'User Group which defines ownership of Configs, Devices and Groups'
    help = group.get('help')
    if not help:
        return 'Permission group, description not available'
    return help


def user_visible_users(user):
    """What other users are visible to the given user

    * Users who are members of an ownership group to which the user belongs

    returns auth.User query
    """
    if constrained_user(user):
        users_groups = ownership_groups_query(user)
        return (
            models.User.objects.order_by('username')
            .filter(Q(groups__in=users_groups) | Q(id=user.id))
            .exclude(is_superuser=True)
            .exclude(is_staff=True)
            .distinct('username')
            .prefetch_related(
                'groups',
            )
        )
    else:
        return models.User.objects.order_by('username')


def user_visible_groups(user):
    """What groups are visible to the given user"""
    if constrained_user(user):
        return user.groups.order_by('name').all()
    else:
        return models.Group.objects.order_by('name').all()


def users_in_group(group):
    """Get a query for "users in this group" """
    return models.User.objects.order_by('username').filter(groups=group)
