import glob
import subprocess
import os
import time
import re
import shutil
from osupdates import decorators
from fussy import nbio
import logging
log = logging.getLogger("database-operations")
BACKUP_EXTENSION = '.sql.gz'
MD5_EXTENSION = BACKUP_EXTENSION + '.md5'
SUDO_PREFIX = ['sudo','-u','postgres','-n']
SQL_COMMAND = SUDO_PREFIX + ['psql','-t','-A','-c']


def cleanup_backups(folder, max_num):
    backup_pattern = os.path.join(folder,
                                  '*' + BACKUP_EXTENSION)
    backup_files = sorted(glob.glob(backup_pattern), reverse=True)
    if len(backup_files) > max_num:
        to_be_removed = backup_files[max_num - 1:]
        for file_to_be_removed in to_be_removed:
            try:
                os.remove(file_to_be_removed)
                os.remove(file_to_be_removed + '.md5')
            except:
                pass

def latest_pgsql():
    import glob
    available = sorted([
        (
            [
                int(y) for y in x.split('-',1)[1].split('.')
            ],
            x
        )
        for x in glob.glob('/usr/pgsql-*')
    ])
    if available:
        return available[-1][1]
    return None


def do_backup(folder, db='burnin', file_prefix='monitoring_server'):
    """Backup the database.

    There are a few tricky things going on besides pg_dump.
    We are using sed to remove the line creating the factory user so as to
    not expose the user for modificaton. We identify the factory user using
    Django user password SHA (only the last, and unique, part to avoid special
    characters).

    MD5 checksum is also created alongside the backup.
    """
    backup_name = '%s_%s%s' % (
        file_prefix,
        time.strftime('%Y-%m-%d-%H%M%S', time.gmtime()),
        BACKUP_EXTENSION,
    )
    filename = os.path.join(folder, backup_name + BACKUP_EXTENSION)
    pg_dump = 'pg_dump'
    latest = latest_pgsql()
    if latest:
        pg_dump = os.path.join(latest, 'bin',pg_dump)

    with decorators.tempdir(prefix='pgbackups-',suffix='-dump') as tempdir:
        tempfile = os.path.join(tempdir, backup_name)
        cmd = '%(pg_dump)s %(db)s | gzip -9 > %(tempfile)s'%locals()
        log.debug("Backup command: %s", cmd)
        nbio.Process( cmd )()
        nbio.Process( 'md5sum %(tempfile)s > %(tempfile)s.md5'%locals())()
        shutil.move( tempfile, folder)
        shutil.move( tempfile+'.md5', folder)
    return os.path.join(folder,os.path.basename(tempfile))

def is_gzip_file(file):
    """Test if this is an actual gzip'd backup file"""
    with open(file,'rb') as fh:
        return fh.read(2) == b'\x1f\x8b'

def create_restore_db(db, user):
    restore_db = '%s-%s'%(db, time.strftime('%Y-%m-%d-%H%M%S', time.gmtime()))
    log.info("Creating restoration DB %s",restore_db)
    for command in [
        ['createdb','-E','UTF-8','-O',user,restore_db]
    ]:
        final_command = ['sudo','-n','-u','postgres'] + command
        nbio.Process(final_command)()
    return restore_db

def sanity_check(db,restore_db):
    # Now, do some basic sanity checks...
    for test,description in [
        ('select count(1) from config_server','No server table'),
        ('select count(1) from config_event','No event table'),
    ]:

        original = nbio.Process(SQL_COMMAND+[test,db])().decode('utf-8').strip()
        try:
            replacement = nbio.Process(SQL_COMMAND+[test,restore_db],stderr=nbio.Process.STDOUT)().decode('utf-8').strip()
        except nbio.ProcessError:
            log.error('Unable to perform sanity check on resulting database, aborting: %s', description)
            return False
    for fix in [
        "update auth_user set is_superuser=FALSE,is_staff=FALSE where username!= 'factory'",
    ]:
        original = nbio.Process(SQL_COMMAND+[fix,restore_db])().decode('utf-8').strip()
    password = nbio.Process(SQL_COMMAND+["select password from auth_user where username='factory'",restore_db])().decode('utf-8').strip()
    nbio.Process(SQL_COMMAND+["update auth_user set password='%s' where username='factory'",restore_db])()
    return True

def shutdown_db_connections(db):
    nbio.Process([
        'sudo','-u','postgres',
        'psql','-c',
        """ALTER DATABASE "%(db)s" CONNECTION LIMIT 1;\
SELECT pg_terminate_backend(pid) FROM pg_stat_activity\
 WHERE datname = '%(db)s' and query not LIKE '%%pg_stat_activity%%';"""%locals(),
        'template1',
    ])()

def run_post_install():
    """Run the post-install operation on the newly fixated database"""
    #nbio.Process(['/opt/firmware/current/.post-install'])()
    log.warning("Triggering firmware post-install now")
    nbio.Process(['systemd-run','--unit','restore-post-install','/opt/firmware/current/.post-install','--direct-log'])()

def restore_backup( file, db='burnin', user='burnin' ):
    """Restore the database in the given file to the given DB"""
    restore_db = create_restore_db(db,user)

    with decorators.tempdir(prefix='pgbackups-',suffix='-restore') as tempdir:
        # os.chdir(tempdir)
        os.chmod(tempdir,0o755)
        filename = os.path.abspath(os.path.join(tempdir,'backup.sql.gz'))
        shutil.copy(file,filename)
        os.chmod(filename,0o644)

        if is_gzip_file(filename):
            nbio.Process('gzip -f --decompress %(filename)s'%locals())()
            files = os.listdir(tempdir)
            if not len(files) == 1:
                log.warning("Expected a single-file backup, aborting")
                raise SystemExit(1)
            filename = os.path.join(tempdir,files[0])
            # TODO: sandbox this
            command = SUDO_PREFIX + ['psql','-f',filename,restore_db]
            nbio.Process(command)()
        else:
            # TODO: sandbox this
            command = SUDO_PREFIX + ['pg_restore','-d',restore_db, filename]
            nbio.Process(command,stderr=nbio.Process.STDOUT)()

    if sanity_check(db,restore_db):
        original_db = '%(restore_db)s_original'%locals()
        log.info("Restore passes sanity check, fixating")
        shutdown_db_connections(db)
        nbio.Process(
            SUDO_PREFIX + ['psql','-c','''alter database "%(db)s" rename to "%(original_db)s"'''%locals()]
        )()
        nbio.Process(
            SUDO_PREFIX + ['psql','-c','''alter database "%(restore_db)s" rename to "%(db)s"'''%locals()]
        )()
        # TODO: if this fails, revert the db and re-run post-install
        # but *this* process will be killed before we get a chance to run...
        run_post_install()
    else:
        log.warning("Removing restoration DB: %s", restore_db)
        nbio.Process(
            SUDO_PREFIX + ['psql','-c',"drop database \"%(restore_db)s\""%locals()]
        )()

def upload_backup_file(file_path, log_path, location, username, password):
    # Build CURL ftp command line
    cmd = [
        'curl',
        '-T "{%(file_path)s,%(file_path)s.md5}"' % locals(),
        'ftp://%(location)s' % locals(),
    ]
    # Add username and password IF provided
    if username and password:
        cmd.append('--user %(username)s:%(password)s' % locals())
    # Add logging
    cmd.append('>> %(log_path)s 2>&1 &' % locals())
    cmd_str = ' '.join(cmd)
    return_code = subprocess.call(cmd_str, preexec_fn=os.setsid, shell=True)
    return return_code
