"""Common components for setting up a docker build in a pre-docker script"""
import logging, os, glob, re
from fussy import nbio, twrite
from osupdates import decorators
from .fabmetadata import run, get_server_links

log = logging.getLogger(__name__)


def rsync_into(source, target):
    """Rsync the source directory's contents into target

    Expands the sources using listdir so even dot-files and
    the like will be copied over...
    """
    ignore = set(('.git', '.gitignore'))
    command = (
        [
            'rsync',
            '-av',
        ]
        + [os.path.join(source, f) for f in os.listdir(source) if f not in ignore]
        + [
            target + '/',
        ]
    )
    log.info("Rsync command: %s", " ".join(command))
    return nbio.Process(command)()


def current_ronin():
    """Pull and unpack the current ronin from releases server"""
    available = []
    for target, date, size in get_server_links('ronin'):
        if target.endswith('.tar.xz') and 'ronin' in target.split('/')[-1]:
            available.append((target, date, size))
    available.sort(key=lambda x: x[1])
    return available[-1][0]


def pull_ronin(
    sku='firmware',
    target='/opt/firmware/current/ronin',
    etc_path='/opt/firmware/current/etc',
):
    link = current_ronin()
    tmp = os.path.expanduser('~/tmp')
    if not os.path.exists(tmp):
        run('mkdir -p %(tmp)s' % locals())
    with decorators.cd(tmp):
        filename = link.split('/')[-1]
        dirname = sku
        if not os.path.exists(filename):
            run('wget --no-check-certificate %(link)s' % locals())
        log.info('Using ronin: %s', filename)
        run('rm -rf %(target)s' % locals())
        ronin_name = filename
        parent = os.path.dirname(target)
        run(
            'tar --strip-components=3 --directory=%(parent)s -xvf %(filename)s'
            % locals()
        )
        filename = os.path.join(etc_path, 'ronin.conf')
        if not os.path.exists(os.path.dirname(filename)):
            run('mkdir -p %(dir)s' % {'dir': os.path.dirname(filename)})
        with open(filename, 'w') as fh:
            fh.write(
                '''[release]
ronin=%(ronin_name)s
'''
                % locals()
            )
        return filename


SUPERVISOR_REPLACER = re.compile(r'digistream[0-9]*')


def fill_supervisor_username(root, username):
    """Fill in the username in supervisor configurations

    root -- should be the directory that will become `/opt/firmware/current` in the final
            image
    username -- username to use for the running daemons in supervisor
    """
    for path in glob.glob(os.path.join(root, 'etc/supervisord.d/*.conf')) + glob.glob(
        os.path.join(root, 'etc/supervisor/conf.d/*.conf')
    ):
        with open(path, 'r') as fh:
            content = fh.read()
        twrite.twrite(path, SUPERVISOR_REPLACER.sub(username, content))


def move_etcfiles(root):
    """Move the etcfiles to the correct directories for centos7

    root -- should be the directory that will become `/opt/firmware/current` in the final
            image
    """
    from osupdates import centos

    class TempCentos(centos.CentOS):
        def sudo(self, *args, **named):
            return self.run(*args, **named)

    t = TempCentos(
        'centos7',
        version='7.3',
        remote=False,
    )
    t.move_supervisor(
        os.path.join(root, 'etc')
    )  # low-level code expects to be pointed to etc subdirectory


def upload_firmware(filename, sku):
    base = os.path.basename(filename)
    target = os.path.join('/var/firmware/media/firmware/', sku, base)
    directory = os.path.dirname(target)
    run('ssh configserver@10.1.0.224 "mkdir -p %s"' % (directory,))
    run('scp %s configserver@10.1.0.224:%s' % (filename, target))
    if os.path.exists(filename + '.release.txt'):
        run(
            'scp %s.release.txt configserver@10.1.0.224:%s.release.txt'
            % (filename, target)
        )
