"""Manage product themes for rebranding"""
import subprocess
from atxstyle.sixishdj import gettext_lazy as _

import os, json, shutil, glob, logging
from django import forms
from django.core import validators
from django.conf import settings
from media import bigfilewidget
from osupdates import decorators
from fussy import nbio

from six.moves.configparser import SafeConfigParser, NoOptionError, NoSectionError
from . import loadoptions, rebootrequired
from django.http import HttpResponseRedirect
from atxstyle.sixish import lru_cache

log = logging.getLogger(__name__)

REMOTE_MEDIA = bool(os.getenv('REMOTE_PROTECTED_HOST'))
STATIC_THEMES = bool(os.getenv('CONTAINER_RUNTIME') == 'kubernetes')


@lru_cache(maxsize=1)
def get_available_themes():
    return [
        (os.path.basename(path), path)
        for path in glob.glob('/opt/firmware/current/themes/*')
    ]


def current_theme():
    if STATIC_THEMES:
        return 'atx'
    try:
        return os.path.basename(os.readlink(loadoptions.THEME_LINK))
    except Exception as err:
        return 'atx'


def theme_from_host(request):
    """Load theme name from host header"""
    host = request.META.get('HTTP_HOST')
    return _theme_from_host(host)


@lru_cache(maxsize=20)
def _theme_from_host(host):
    theme = None
    if host:
        theme = host.split('.')[0]
        available = dict(get_available_themes())
        available['atx'] = 'default'
        if theme not in available and not theme.isdigit():
            log.info('Theme %s not in available: %s', theme, sorted(available.keys()))
            theme = current_theme()
    log.info('Host %s => theme %r', host, theme)
    return theme


class FactoryThemeForm(forms.Form):
    """Form to reconfigure the theme"""

    theme = forms.ChoiceField(
        label=_("Theme"),
        required=True,
        choices=[
            ('atx', 'ATX (Default)'),
        ]
        + [(name, name) for name, path in get_available_themes()],
        initial=current_theme(),
        help_text=_("Product Branding Theme to enable"),
    )
    permissions = ['factory']

    def __init__(self, *args, **named):
        if 'instance' in named:
            del named['instance']
        super(FactoryThemeForm, self).__init__(*args, **named)
        # Just after we've updated...
        self.fields['theme'].initial = current_theme()

    def save(self):
        theme = self.cleaned_data.get('theme')


EXPECTED = [
    'css',
    'img/company-logo.svg',
    'img/logo.svg',
    'conf/*.conf',
]
CONFIG_KEYS = [
    'company_name',
    'company_short_name',
    'company_legal_name',
    'product_url_name',
    'product_family',
    'sku',
    'sku_name',
    'themed',
    'support_url',
    'encoder_only',
    'encode_video',
    'encode_audio',
    'encode_text',
    'world_is_node',
]
DEFAULT_THEME = {
    'company_name': 'ATX Networks',
    'company_short_name': 'ATX',
    'company_legal_name': 'ATX Networks Inc.',
    'product_url_name': settings.PRODUCT.lower(),
}
for _key in CONFIG_KEYS:
    _setting = getattr(settings, _key.upper(), None)
    if _setting is not None:
        DEFAULT_THEME[_key.lower()] = _setting


def load_theme_config_by_name(theme):
    """Load the given theme configuration with product overrides and defaults"""
    if theme == 'atx':
        return DEFAULT_THEME.copy()
    theme_configs = sorted(
        glob.glob(os.path.join(loadoptions.THEME_DIRECTORY, theme, 'conf', '*.conf'))
    )
    theme_configs += sorted(
        glob.glob(
            os.path.join(
                loadoptions.THEME_DIRECTORY,
                '%s-%s'
                % (
                    settings.PRODUCT,
                    theme,
                ),
                'conf',
                '*.conf',
            )
        )
    )
    config = None
    if theme_configs:
        config = SafeConfigParser()
        config.read(theme_configs)
    result = {}
    for key in CONFIG_KEYS:
        value = None
        if config:
            try:
                value = config.get('product', key)
            except (NoOptionError, NoSectionError) as err:
                pass
        if value is None:
            value = default_config_key(key)
        result[key] = value
    theme_json_file = os.path.join(
        loadoptions.THEME_DIRECTORY,
        '%s-%s'
        % (
            settings.PRODUCT,
            theme,
        ),
        'theme.json',
    )
    if os.path.exists(theme_json_file):
        result['theme_json'] = json.loads(open(theme_json_file).read())
    return result


def default_config_key(key):
    """Given key, use default/gateway lookup to find the value"""
    value = getattr(settings, key.upper(), None)
    if value is None:
        if key == 'themed':
            value = loadoptions.getboolean('product', 'themed', False)
        elif key == 'support_url':
            value = loadoptions.getstring('product', key, 'https://atx.com/support')
        elif key in DEFAULT_THEME:
            value = DEFAULT_THEME[key]
    return value


@lru_cache(maxsize=8)
def config(theme=None):
    """Populate the `config.Config` JS external in webpack

    Use it in JS via::

        import Config from 'config';

        `Product is: ${Config.sku_name}`

    Included via the `appconfig.appconfig` html template tag
    in the `base.html` template.
    """
    from django.conf import settings

    result = {
        'world_is_node': getattr(
            settings,
            'WORLD_IS_NODE',
            True,
        ),
        'top_navigation': getattr(settings, 'GUI_TOP_NAVIGATION', False),
        'white_cards': getattr(settings, 'GUI_WHITE_CARDS', False),
    }
    if theme is not None:
        # Load from the named theme rather than settings...
        result.update(load_theme_config_by_name(theme))
    else:
        for key in CONFIG_KEYS:
            result[key] = default_config_key(key)
    return result


def update_themed_user_passwords(company, product):
    """Update each user currently having a non-themed password"""
    from . import basepermissions
    from django.contrib.auth import models as auth_models

    for user in auth_models.User.objects.all():
        old_password = basepermissions.default_password(user.username)
        if user.is_superuser:
            continue
        if user.check_password(old_password):
            new_password = basepermissions.default_password(
                user.username,
                company=company,
                product=product,
            )
            log.info("Updating password for user %s to %s", user.username, new_password)
            user.set_password(new_password)
            user.save()
        else:
            log.info("%s did not have password %s", user.username, old_password)


def update_themed_hostnames(product):
    """Update each interface to have product-specific dhcp_hostnames

    Gateway only...
    """
    from netconfig import models as netconfig_models

    netconfig_models.Interface.objects.all().update(
        dhcp_hostname=product.lower().replace(' ', ''),
    )


def set_theme(theme):
    """Unpack theme into /var/firmware/product-theme/"""
    if not STATIC_THEMES:
        create_theme_link(theme)
        loadoptions.options = None
        loadoptions.loadoptions()
        rebootrequired.set('Theme Selected')
        update_themed_user_passwords(
            loadoptions.getstring('product', 'company_short_name'),
            loadoptions.getstring('product', 'product_family'),
        )
        update_themed_hostnames(loadoptions.getstring('product', 'product_family'))
    else:
        log.warning("Attempted to set theme on STATIC_THEMES product")


def theme_path(name):
    for key, path in get_available_themes():
        if key == name:
            return path
    raise ValueError("Unknown theme name: %s" % (name,))


def create_theme_link(name):
    if STATIC_THEMES:
        raise RuntimeError("Trying to set theme on STATIC_THEMES product")
    link = loadoptions.THEME_LINK
    if os.path.exists(link):
        os.remove(link)
    if not os.path.exists(loadoptions.THEME_DIRECTORY):
        os.makedirs(loadoptions.THEME_DIRECTORY)
    if REMOTE_MEDIA:
        # have to copy theme to media...
        if name not in ('atx', None):
            THEME_COPY = os.path.join(loadoptions.THEME_DIRECTORY, name)
            # CVEC-83 Can't check the call because of azure permissions issues...
            subprocess.call(
                [
                    'rsync',
                    '-av',
                ]
                + glob.glob(
                    theme_path(name) + '/*',
                )
                + [
                    THEME_COPY + '/',
                ]
            )
            os.symlink('./' + name, link)
    else:
        if name not in ('atx', None):
            os.symlink(theme_path(name), link)


def theme_url(request, remainder):
    """Get the theme-specific URL for a given request"""
    theme = theme_from_host(request)
    remainder = remainder.lstrip('/')
    if theme and theme != 'atx':
        if STATIC_THEMES:
            return HttpResponseRedirect(
                '/static/product-theme/%s/%s'
                % (
                    theme,
                    remainder,
                )
            )
        else:
            # gateway mode with uploadable themes
            return HttpResponseRedirect(
                '/media/product-theme/%s/%s'
                % (
                    theme,
                    remainder,
                )
            )
    else:
        return HttpResponseRedirect('/static/%s' % (remainder,))


from atxstyle.sixishdj import gettext as _
