"""Provides file-system-storage to settings.PROTECTED_DIR

Use `with_protected_direct` and `protected_redirect` for 
sending users to the `/protected` URL. 3 modes, local,
single-host-nginx and remote-host-nginx modes...
"""
from atxstyle.sixish import unicode
from django.http import (
    HttpResponse,
    HttpResponseForbidden,
    StreamingHttpResponse,
)
from django.conf import settings
from django.core.files.storage import FileSystemStorage
from django.db.models.fields.files import FieldFile
from functools import wraps
from glob import glob
from django.http.response import HttpResponseRedirect
from django.urls import reverse
import logging, os

log = logging.getLogger(__name__)

if not settings.PROTECTED_URL.endswith('/'):
    log.warning('PROTECTED_URL should end with a / character')
    settings.PROTECTED_URL += '/'


def with_protected_redirect(
    content_type='application/json',
    base_dir=settings.PROTECTED_DIR,
    base_url=settings.PROTECTED_URL,
    filename=None,
):
    """Decorate a function for redirection to nginx in base_dir

    function must return a str/unicode filename, any other value is
    returned unchanged...
    """

    def _with_protected_redirect(function):
        @wraps(function)
        def wrapped(request, *args, **named):
            return protected_redirect(
                function(request, *args, **named),
                request,
                content_type=content_type,
                base_dir=base_dir,
                base_url=base_url,
                filename=filename,
            )

        return wrapped

    return _with_protected_redirect


def protected_redirect(
    result,
    request,
    content_type='application/json',
    base_dir=settings.PROTECTED_DIR,
    base_url=settings.PROTECTED_URL,
    filename=None,
):
    """Core operation of with_protected_redirect decorator

    Performs NGINX X-Accel-Redirect iff HTTP_X_NGINX_HOSTED header present
    """
    if isinstance(result, FieldFile):
        # use the url directly...
        url = result.url
        if not url.startswith(base_url):
            return HttpResponseForbidden(url)

        def opener():
            result.open('rb')
            return result

        relative = url

    elif not isinstance(result, (bytes, unicode)):
        return result
    else:
        relative = os.path.relpath(result, base_dir)
        if relative.startswith('../') or relative.startswith('/'):
            return HttpResponseForbidden(relative)
        url = base_url + relative

        def opener():
            return open(os.path.join(base_dir, relative), 'rb')

    if getattr(settings, 'REMOTE_PROTECTED_HOST', False):
        # Protected Host that will call back to us to verify that we have granted
        # this session access to the URL above (only allows for a single access
        # as the view will pop the URL from the auth table)
        final_url = base_url + relative
        log.info('Issuing remote protected redirect for %s', final_url)
        request.session.setdefault('PROTECTED_URL_AUTH', {})[final_url] = True
        response = HttpResponseRedirect(final_url)
        return response

    if request.META.get('HTTP_X_NGINX_HOSTED'):
        response = HttpResponse('', content_type=content_type)
        relative = url
        response['X-Accel-Redirect'] = relative
    else:
        response = StreamingHttpResponse(opener(), content_type=content_type)
    if filename:
        if isinstance(filename, unicode):
            filename = filename.encode('utf-8')
        elif relative:
            filename = os.path.basename(relative)
        elif url:
            filename = os.path.basename(filename)
        else:
            filename = None
        if filename:
            response['Content-Disposition'] = b'attachment; filename="%s"' % (filename,)
    return response


class ProtectedStorage(FileSystemStorage):
    def url(self, name):
        return settings.PROTECTED_URL + name

    def deconstruct(self):
        return (
            '%s.%s' % (self.__class__.__module__, self.__class__.__name__),
            (),
            {'location': self.location},
        )

    def __eq__(self, other):
        if isinstance(other, (str, unicode)):
            return self.location == other
        else:
            return self.location == getattr(other, 'location', None)


PROTECTED_STORAGE = ProtectedStorage(location='/var/firmware/protected')
