"""Perform automated raid recovery if/when possible..."""
from __future__ import unicode_literals
import logging, os, re, time
from atxstyle import standardlog, humanize
from atxstyle.sixish import as_unicode
from fussy import nbio
log = logging.getLogger(__name__)

DIGITS = '0123456789'
MDADM = '/sbin/mdadm'
PERSONALITY = re.compile(r'[\[](?P<name>[^\]]+)[\]]')
RECOVERY = re.compile(r'[ ]*.*(recovery|resync)[ ]*=[ ]*(?P<progress>[0-9.]+[%]).*finish[ ]*=[ ]*(?P<finish>[^ ]+)')
STRUCTURE = re.compile(r'[ ]*\d+[ ]+blocks.*[\[](?P<total_devices>\d+)[/](?P<current_devices>\d+)[\]][ ]*[\[](?P<upmask>[_U]+)[\]]')

def device_name(name):
    if not name.startswith('/dev/'):
        name = '/dev/'+name
    return name
def drive_from_partition(name):
    return device_name(device_name(name).rstrip(DIGITS))

def raid_status():
    try:
        content = open('/proc/mdstat').read()
        return parse_mdstat(content)
    except Exception:
        return {
            'devices':[], 
            'error':True, 
        }

def parse_mdstat(content):
    structure = {}
    lines = as_unicode(content).strip().splitlines()
    structure['personalities'] = PERSONALITY.findall(lines[0])
    structure['unused'] = [
        device for device in lines[-1].strip().split(':', 1)[1].split()
        if device != '<none>'
    ]
    failures = 0
    recoveries = 0
    devices = structure['devices'] = []
    current = {}
    for line in lines[1:-1]:
        line = line.strip()
        if not line:
            devices.append(current)
            current = {}
        elif not current:
            name, rest = [x.strip() for x in line.split(':', 1)]
            current['name'] = name 
            status, personality, members = rest.split(None, 2)
            current['status'] = status 
            current['personality'] = personality
            current['devices'] = [
                {
                    'name':x.split('[')[0], 
                    'failure': x.endswith('(F)'), 
                }
                for x in members.split()
            ]
            failures += len([d for d in current['devices'] if d.get('failure')])
        else:
            for matcher in (RECOVERY, STRUCTURE):
                match = matcher.match(line)
                if match:
                    if matcher is RECOVERY:
                        current['recovery'] = match.groupdict()
                        recoveries += 1
                    else:
                        def as_int(x):
                            try:
                                return int(x, 10)
                            except ValueError:
                                return x
                        current.update(dict([
                            (k, as_int(v))
                            for k, v in match.groupdict().items()
                        ]))
                        if current['total_devices'] > current['current_devices']:
                            current['degraded'] = True 
                        else:
                            current['degraded'] = False
    structure['failures'] = failures 
    structure['recoveries'] = recoveries
    return structure


def recovery_can_proceed(status):
    """Return bool(can_proceed), [messages]"""
    can_proceed = False
    messages = []
    if not status.get('failures'):
        messages.append("There are no currently recognized faults, cannot proceed")
    elif status.get('recoveries'):
        messages.append("There is a rebuild/recovery in progress, cannot proceed")
    else:
        can_proceed = True 
    if can_proceed:
        to_rebuild = devices_to_rebuild(status)
        if len(to_rebuild) > 1:
            can_proceed = False 
            messages.append(
                "%(count)s devices have failed (%(names)s), we can only automatically recover from a single device failure"%{
                'count':len(to_rebuild), 
                'names': ',  '.join(sorted(to_rebuild)), 
            })
    if can_proceed:
        for array, partitions in partitions_to_remove(status):
            if len(partitions)>1:
                messages.append('More than one partition would be removed from %s', array['name'])
                can_proceed = False 
    return can_proceed, messages

def devices_to_rebuild(status):
    to_rebuild = set()
    for device in status['devices']:
        for sub in device['devices']:
            if sub.get('failure'):
                to_rebuild.add(sub['name'].rstrip(DIGITS))
    return to_rebuild
def partitions_to_remove(status, to_rebuild=None):
    """Which partitions should be removed to rebuild set of drives in to_rebuild"""
    if to_rebuild is None:
        to_rebuild = devices_to_rebuild(status)
    assert len(to_rebuild) == 1
    for array in status['devices']:
        subs = []
        for sub in array['devices']:
            if sub['name'].rstrip(DIGITS) in to_rebuild:
                subs.append(sub)
        yield array, subs

if os.getuid() != 0:
    def run_command(command):
        if isinstance(command, list):
            log.info('Would Run: %s', ' '.join(command))
        else:
            log.info('Would Run: %s', command)
else:
    def run_command(command):
        if isinstance(command, list):
            log.info('Running: %s', ' '.join(command))
        else:
            log.info('Running: %s', command)
        output = as_unicode(nbio.Process(command)())
        return output


@standardlog.with_debug( 'raid-recovery', product='firmware', clear=False, do_console=True )
def remove_and_shutdown():
    """Main operation to trigger removal of failed disks and shutdown chassis"""
    time.sleep(5)
    status = raid_status()
    if not status.get('devices'):
        log.error('No RAID devices detected')
        return 2
    can_proceed, why_not = recovery_can_proceed(status)
    if not can_proceed:
        for message in why_not:
            log.warning('%s', message)
        return 2
    to_rebuild = devices_to_rebuild(status)
    assert len(to_rebuild) == 1
    
    removed = 0
    for array, subs in partitions_to_remove(status, to_rebuild):
        for sub in subs:
            log.warning('Removing %s from %s', sub['name'], array['name'])
            command = [
                MDADM, '--manage', 
                    device_name(array['name']), 
                    '--fail', device_name(sub['name']), 
            ]
            run_command(command)
            command = [
                MDADM, '--manage', 
                    device_name(array['name']), 
                    '--remove', device_name(sub['name']), 
            ]
            run_command( command )
            removed += 1
    
    command = ['shutdown', '-h', 'now']
    run_command( command )

DISK_SIZE = re.compile(r'^Disk (?P<name>[/]dev[/]sd\w+)[:].*?(?P<size>\d+)[ ]+bytes$' )
def disk_sizes():
    content = as_unicode( nbio.Process(['fdisk','-l'])() )
    return parse_disk_sizes(content)
def parse_disk_sizes(content):
    """Parse output of fdisk -l to pull out whole-disk byte sizes"""
    mapping = {}
    for line in content.strip().splitlines():
        match = DISK_SIZE.match(line)
        if match:
            device,size = match.group('name'),match.group('size')
            size = int(size)
            disk = drive_from_partition(device)
            mapping.setdefault(disk, {
                'name':disk,
                'size': size,
                'size_formatted': humanize.human_bytes(size),
            })
    return mapping

@standardlog.with_debug('raid-recovery', product='firmware', clear=False, do_console=True )
def recovery():
    """Entry point for searching for and automatically adding a replacement drive
    
    Replacement drive *must* be un-partitioned and empty,
    as we don't want to risk killing a running drive, so when you 
    plug in the device we're going to check that and error out.
    """
    # first problem is that we need to *find* the disk,
    # likely is in /dev/sd? but which one? Well, it *can't* be a 
    # device which is currently in our raid arrays.
    # It *must* be a disk with enough space to hold our partitions,
    # and we assume that it has been partitioned with at least as 
    # much space as we need (note: we don't currently deal with 
    # un-partitioned drives)
    status = raid_status()
    active_devices = set()
    for array in status['devices']:
        for sub in array['devices']:
            active_devices.add(drive_from_partition(sub['name']))
    if len(active_devices)>1:
        log.info('Have more than one active disk, cannot proceed')
        return 0
    elif not active_devices:
        log.warning('No active RAID devices detected, either a non-raid config or all disks have failed' )
        return 0
    active = list(active_devices)[0]
    # now look for another device that is *not* part of the raid arrays but *is*
    # a disk and is large enough to hold our partitions...
    sizes = disk_sizes()
    if not active in sizes:
        log.error('Could not retrieve current partition size for %s', active)
        return 3
    current_size = sizes[active]
    target_disk = None
    for other in sizes.values():
        if other['name'] == active:
            continue 
        elif other['size'] >= current_size['size']:
            target_disk = other 
        else:
            log.info('Disk %s size %s is not large enough', other['name'], other['size_formatted'])
    if not target_disk:
        # TODO: 
        log.error("Could not find a disk with at least %s", current_size['size_formatted'])
        return 4
    # okay, so we have a target disk and we've decided it is large enough...
    log.info('Selected %s as target disk, partitioning', target_disk['name'])
    run_command('sfdisk -d %s | sfdisk -f %s'%(active, target_disk['name']))
    for array in status['devices']:
        for sub in array['devices']:
            target = device_name(sub['name']).replace( 
                drive_from_partition(sub['name']), 
                device_name(target_disk['name']), 
            )
            log.info('Adding %s to %s', target, array['name'])
            run_command([MDADM, device_name(array['name']), '--add', target])
    run_command(['grub-install',drive_from_partition(target_disk['name'])])
    log.info('Finished')
    return 0
