from __future__ import absolute_import, unicode_literals
from atxstyle.sixishdj import gettext_lazy as _


assert _
import os
import logging
from django.conf import settings
from django.contrib.auth.forms import AuthenticationForm
from django.contrib.auth import models as auth_models
from django.core.exceptions import PermissionDenied
from django.contrib.auth import models, login
from django.apps import apps
from django import forms as django_forms
from . import users, forms, system, ownership

log = logging.getLogger(__name__)
System = apps.get_model('config', 'System')

from media.uploads import Uploads
from reactforms import (
    models as reactforms_models,
    forms as reactforms_forms,
    tabular,
    formview,
)
from atxstyle import models as atx_models, users as atx_users
from . import producttheme

AuthenticationForm.permissions = []


REGISTRY = reactforms_forms.FormRegistry()


class LoginView(reactforms_forms.FormView):
    permissions = []

    def create_get_form(self, request, record):
        return AuthenticationForm()

    def create_post_form(self, request, record):
        post = self.get_request_post(request)
        form = AuthenticationForm(request, post)
        return form

    def handle_save(self, request, record, form):
        login(request, form.user_cache)
        return {'success': True, 'messages': []}


class UserEditView(reactforms_forms.FormView):
    permissions = []  # handled internally

    def check_permissions(self, request, record, form=None, method=None):
        if 'delete-instance' in request.GET or 'delete-instance' in request.POST:
            if not request.user.has_perm('auth.delete_user'):
                raise PermissionDenied(
                    "Account %s does not have permission to perform this action"
                    % (request.user.username,)
                )

            if not atx_models.is_user_deletable(record):
                raise PermissionDenied("User %s cannot be deleted" % (record.username,))
        return super(UserEditView, self).check_permissions(
            request, record, form, method=method
        )

    def create_get_form(self, request, record):
        return users.actor_edit_form(request.user, record)(
            initial=request.GET, instance=record if record else None
        ).for_actor(request.user)

    def create_post_form(self, request, record):
        post = self.get_request_post(request)
        return users.actor_edit_form(request.user, record)(
            post, instance=record if record else None
        ).for_actor(request.user)


class SystemPowerActionsView(reactforms_forms.FormView):
    def handle_post(self, request, record=None):
        return system.process_reboot_request_json(request)


REGISTRY.add(
    LoginView(
        key='login',
        form=AuthenticationForm,
        model=models.User,
        permissions=[],
    )
)
REGISTRY.add(
    UserEditView(
        key='User',
        form=None,
        model=models.User,
        permissions=[],
        # debug=True,
    )
)
REGISTRY.register(
    'Group',
    users.GroupForm,
    models.Group,
    # debug=True,
)
REGISTRY.register(
    'ClearEventLog',
    forms.EventLogClearForm,
    System,
    permissions=['config.debugging'],
)


def lookup_user_preference(request, type_key, *args, **named):
    prefs = reactforms_models.UserPref.objects.filter(
        username=request.user.username
    ).first()
    if not prefs:
        prefs = reactforms_models.UserPref.objects.create(
            username=request.user.username,
        )
    return prefs


REGISTRY.register(
    'UserPref',
    reactforms_forms.UserPrefForm,
    reactforms_models.UserPref,
    lookup_record=lookup_user_preference,
    owner_field='user',
)
REGISTRY.register(
    'Dashboard',
    reactforms_forms.DashboardForm,
    reactforms_models.Dashboard,
)
# REGISTRY.register(
#     'GroupDashboards',
#     reactforms_forms.GroupDashboardsForm,
#     reactforms_models.GroupDashboards,
# )

REGISTRY.register('EULAForm', forms.EULAForm, System, permissions=[])
REGISTRY.register(
    'LocationForm', forms.LocationForm, System, permissions=['config.location']
)
REGISTRY.register('DateForm', forms.DateForm, System, permissions=['config.location'])

REGISTRY.register(
    'SerialForm',
    forms.SerialForm,
    System,
    permissions=['factory'],
)
REGISTRY.register(
    'ProductTheme',
    producttheme.FactoryThemeForm,
    System,
    permissions=['factory'],
)
REGISTRY.register(
    'PeriodicRebootForm',
    forms.PeriodicRebootForm,
    System,
    permissions=['config.shutdown'],
)
REGISTRY.register(
    'LicenseClientForm',
    forms.LicenseClientForm,
    System,
    permissions=['config.licensing'],
)


def verified_licenses(request, type_key=None, id=None):
    try:
        import json

        content = json.loads(open('/etc/atxlicense/current.json').read())
        content['success'] = True
    except Exception:
        content = {
            'error': True,
            'messages': ['Unable to read the current licensing metadata'],
            'verified': [],
        }
    return content


REGISTRY.add(
    formview.FunctionWrapperView(
        key='LicenseClientLicenses',
        function=verified_licenses,
        view_permissions=['config.view_system'],
        permissions=['config.licensing'],
    )
)

REGISTRY.register(
    'SNMPForm',
    forms.SNMPForm,
    System,
    permissions=['config.snmp'],
    # debug=True,
)
REGISTRY.register(
    'FirmwareForm',
    forms.FirmwareForm,
    System,
    permissions=['config.upgrade'],
)
REGISTRY.add(
    SystemPowerActionsView(
        'SystemPowerActions',
        form=forms.RebootForm,
        model=System,
        permissions=['config.shutdown'],
    )
)


def base_users(request, type_key):
    if not request.user.is_authenticated:
        log.info("An un-authenticated user is requesting User listing")
        raise PermissionDenied("You must be logged in to request a user listing")
    return ownership.user_visible_users(request.user)


def annotate_users(response, request, table):
    for user in response['Users']:
        if user['username'] == request.user.username:
            user['editable'] = True
        elif request.user.has_perm('auth.create_user'):
            user['editable'] = True
    return response


# def user_json(record, request, type_key):
#     if isinstance(record, models.User):
#         return atx_models.user_schedule_json_base(record)
#     raise TypeError('Linearising something not a user in the user query')


REGISTRY.add(
    tabular.TabularView(
        key='Users',
        model=models.User,
        form=users.UserTableViewForm,
        default_order=[
            'username',
        ],
        base_query=base_users,
        allowed_traversals=[
            'groups__groupname',
        ],
        annotate_results=annotate_users,
        prefech_related=['groups', 'permissions', 'groups__permissions'],
        view_permissions=[],
        permissions=['auth.edit_users'],
        debug=True,
        # record_json=user_json,
    )
)
models.User.__json__ = atx_models.user_schedule_json


REGISTRY.add(
    tabular.TabularView(
        key='Dashboards',
        model=reactforms_models.Dashboard,
        form=reactforms_forms.DashboardForm,
        prefetch_related=['owner'],
        default_editing_forms=[
            'Dashboard',
        ],
        default_order=[
            'slug',
        ],
        # base_filters=only_group_members,
        view_permissions=['config.view_dashboards'],
        permissions=['config.edit_dashboards'],
    )
)


react_form = REGISTRY.react_form

from atxstyle.sixishdj import gettext as _

assert _
