"""Operation to rebase a model class during migrations"""
from django.db import models
from django.db.migrations.operations.models import ModelOptionOperation


class SetModelBasesOperation(ModelOptionOperation):
    """Resets a model's base classes during migrations so that migration before uses old layout and migration after uses new

    Add it to a migration at the point where the code was modified to
    use the new model bases.

    operations = [
        ...,
        SetModelBasesOperation('modelname',(models.Model,),(models.IntermediateModel,))
        ...,
    ]
    """

    def __init__(self, model_name, old_bases, new_bases):
        super(SetModelBasesOperation, self).__init__(model_name)
        self.old_bases = old_bases
        self.new_bases = new_bases

    def deconstruct(self):
        """Allow the operation to be stored/restored"""
        kwargs = {
            'model_name': self.name,
            'old_bases': self.old_bases,
            'new_bases': self.new_bases,
        }
        return (self.__class__.__qualname__, [], kwargs)

    def state_forwards(self, app_label, state):
        model_state = state.models[app_label, self.name_lower]
        model_state.bases = self.new_bases
        state.reload_model(app_label, self.name_lower, delay=True)

    def state_backwards(self, app_label, state):
        model_state = state.models[app_label, self.name_lower]
        model_state.bases = self.old_bases
        state.reload_model(app_label, self.name_lower, delay=True)

    def database_forwards(self, app_label, schema_editor, from_state, to_state):
        pass

    def database_backwards(self, app_label, schema_editor, from_state, to_state):
        pass

    def describe(self):
        return "Update base classes for the model %s" % self.name

    @property
    def migration_name_fragment(self):
        return 'update_%s_bases' % self.name_lower
