from typing import get_type_hints
import redis, json, os, logging, time
from functools import wraps
from contextlib import contextmanager

log = logging.getLogger(__name__)


def get_channel():
    return redis.from_url(os.getenv('REDIS_URL', 'redis://localhost'))


@contextmanager
def redis_connection():
    """Context manager with a redis connection"""
    redis = get_channel()
    yield redis
    redis.close()


def with_redis_connection(function):
    @wraps(function)
    def with_redis(*args, **named):
        with redis_connection() as redis:
            return function(redis, *args, **named)

    return with_redis


@with_redis_connection
def redis_send(redis, channel, content={}, no_warning=False):
    if not isinstance(content, (bytes, str)):
        content = json.dumps(content)
    if (not no_warning) and not channel.startswith('p.'):
        log.error('Publishing to a channel not in the p.* namespace: %s', channel)
    redis.publish(channel, content)


@with_redis_connection
def redis_stream(redis, channel, content={}):
    """Add content to the given stream with expected {r:json.dumps(content)} operation"""
    if not isinstance(content, (bytes, str)):
        content = json.dumps(content)
    if not channel.startswith('q.'):
        log.error('Xadding to a queue not in the q.* namespace: %s', channel)
    redis.xadd(channel, {'r': content}, maxlen=100, approximate=True)


@contextmanager
def with_redis_lock(lock_name, max_duration=60 * 60):
    LOCK_TEMPLATE = 'locks/%s'
    lock_key = LOCK_TEMPLATE % lock_name
    import time, secrets

    timeout = time.time() + max_duration
    id = secrets.token_hex(16)
    with redis_connection() as redis:
        result = redis.setnx(lock_key, id)
        if result:
            try:
                yield redis
            finally:
                log.info("Lock released")
                redis.delete(lock_key)
        else:
            log.info("We did not acquire the lock, waiting for holder")
            while time.time() < timeout:
                current = redis.get(lock_key)
                if not current:
                    break
                time.sleep(1)
            yield None


_tagged_log_formatter = logging.Formatter(
    datefmt='%Y-%m-%d %H:%M:%S',
)


def send_tagged_log(
    logname,
    message,
    *subs,
    level=logging.INFO,
    function=None,
    line=None,
    module=None,
    resource_id=None,
    subresource_id=None,
    service_id=None,
    stream_id=None
):
    """Directly send a log into redis for the given log name with given tags

    Generally you should use the `atxstyle.standardlog.LoggerAdapter` if you
    are going to send more than a one-off log message.
    """
    message = _tagged_log_formatter.format(
        logging.LogRecord(
            logname,
            level=level,
            pathname=module,
            lineno=line,
            msg=message,
            args=subs,
            func=function,
            exc_info=None,
        )
    )
    redis_stream(
        'q.log',
        {
            'name': logname,
            'msg': message,
            'created': time.time(),
            'funcName': function,
            'pathname': '',
            'filename': '',
            'lineno': line,
            'args': [],
            'levelno': level,
            'service_id': service_id,
            'stream_id': stream_id,
            'resource_id': resource_id,
            'subresource_id': subresource_id,
        },
    )
