"""Trivial non-thread-safe request cache

middleware.SettingsMiddleWare does the actual clearing and setting up...
"""
import contextlib
import threading
import logging
from django.core.signals import request_finished, request_started

log = logging.getLogger(__name__)
current_thread = (
    threading.current_thread
    if hasattr(threading, 'current_thread')
    else threading.currentThread
)

CACHE = {}


def inject(**named):
    thread = current_thread()
    if CACHE.get(thread) is None:
        CACHE[thread] = {}


def clean(**named):
    thread = current_thread()
    try:
        del CACHE[thread]
    except KeyError:
        pass


def thread_cache():
    thread = current_thread()
    return CACHE.get(thread)


request_started.connect(inject)
request_finished.connect(clean)


def set(key, value):
    cache = thread_cache()
    if cache is not None:
        #        log.info("Store: %s", key )
        cache[key] = value
    #    else:
    #        log.warning("Requested caching when there's no cache")
    return value


def get(key, default=None):
    cache = thread_cache()
    if cache is not None:
        #        if key in cache:
        #            #log.info('CACHE hit on %s',key)
        #            pass
        #        else:
        #            log.info('CACHE missing on %s', key )
        return cache.get(key, default)
    return default


def memoize(function):
    """Store the result of this function until the cache is cleared"""
    from functools import wraps

    @wraps(function)
    def cached(*args, **named):
        key = (function, args)
        current = get(key)
        if current is not None:
            return current
        return set(key, function(*args, **named))

    cached.__original__ = function
    return cached


@contextlib.contextmanager
def with_cache():
    inject()
    yield
    clean()
