import logging, os, sys
import logging.handlers
import socket
from functools import wraps
from traceback import format_exception
import io

try:
    from fussy.jsonlog import (
        JSONSocketHandler,
        LOGGING_PORT as JSON_LOGGING_PORT,
        default_tags,
    )
except ImportError as err:
    JSONSocketHandler = None
    JSON_LOGGING_PORT = 2514
JSON_LOGGING_HOST = os.getenv('JSON_LOGGING_HOST') or 'localhost'

DEFAULT_ROTATION_SIZE = 1024 * 1024 * 10
urllib3 = None
try:
    from requests.packages import urllib3
    from requests.packages.urllib3.exceptions import InsecureRequestWarning
except ImportError:
    pass
else:
    if hasattr(urllib3, 'disable_warnings'):
        urllib3.disable_warnings(InsecureRequestWarning)

ENABLED = False


class LoggerAdapter(logging.LoggerAdapter):
    def process(self, message, kwargs):
        extra = kwargs.setdefault('extra', {})
        extra.update(self.extra)
        return message, kwargs


def add_console(root):
    console = logging.StreamHandler()
    console.setFormatter(FULL_FORMATTER)
    root.addHandler(console)
    return console


def add_syslog(root, address='/dev/log'):
    try:
        syslog = logging.handlers.SysLogHandler(address=address)
        syslog.setLevel(logging.WARN)
        syslog.setFormatter(SYSLOG_FORMATTER)
    except socket.error:
        # it *was* there, but is no longer there...
        pass
    root.addHandler(syslog)


def get_json_log_tags():
    base = default_tags()
    for key, default in [
        ('resource_id', 0),
        ('subresource_id', 0),
        ('service_id', 0),
        ('stream_id', 0),
    ]:
        if key not in base:
            base[key] = default
    return base


def add_jsonsock(root, host=JSON_LOGGING_HOST, port=JSON_LOGGING_PORT):
    """Add a json socket logger to the root"""
    if JSONSocketHandler:
        json = JSONSocketHandler(
            host=host,
            port=port,
            tags=default_tags(),
        )
        json.setLevel(logging.WARN)
        root.addHandler(json)


def add_crashlogger():
    crash_logger = logging.getLogger('crash')

    def on_crash(typ, value, traceback):
        try:
            try:
                tb = format_exception(typ, value, traceback, limit=10)
            except Exception as err:
                sys.stderr.write('Unable to format %s exception' % (typ.__name__,))
            else:
                sys.stderr.writelines(tb)
                crash_logger.error(''.join(tb))
        except Exception as err:
            crash_logger.error("Unable to format exception: %s", err.__class__)

    sys.excepthook = on_crash


FULL_FORMAT = '%(levelname)- 7s %(name)-30s:%(lineno)- 4d %(asctime)s  %(message)s'
FULL_FORMATTER = logging.Formatter(FULL_FORMAT, datefmt='%Y-%m-%d %H:%M:%S')
SYSLOG_FORMAT = '%(name)s %(message)s'
SYSLOG_FORMATTER = logging.Formatter(SYSLOG_FORMAT)


def enable(do_console=True, do_jsonlog=True):
    global ENABLED
    if not ENABLED:
        ENABLED = True
        root = logging.getLogger('')
        del root.handlers[:]
        if do_console:
            add_console(root)
        do_syslog = True
        try:
            if hasattr(sys.stdout, 'fileno') and os.isatty(sys.stdout.fileno()):
                do_syslog = False
        except io.UnsupportedOperation:
            pass
        if not os.path.exists('/dev/log'):
            do_syslog = False
        if not do_syslog:
            root.setLevel(logging.DEBUG)
        else:
            root.setLevel(logging.INFO)
            add_syslog(root, '/dev/log')
        if do_jsonlog and JSONSocketHandler:
            add_jsonsock(root)

        add_crashlogger()

        logging.getLogger('scheduler.control').setLevel(logging.INFO)
        logging.getLogger('nose.plugins.manager').setLevel(logging.WARNING)
        logging.getLogger('nose_cov').setLevel(logging.WARNING)
        logging.getLogger('py.warnings').setLevel(logging.ERROR)


def rotating_logfile(filename, clear=False, size=DEFAULT_ROTATION_SIZE):
    """Create a full-formatter rotating log-file handler"""
    try:
        os.makedirs(os.path.dirname(filename))
    except OSError:
        pass
    if clear and os.path.exists(filename):
        os.remove(filename)
    handler = logging.handlers.RotatingFileHandler(
        filename,
        maxBytes=size,
        backupCount=1,
    )
    handler.setFormatter(FULL_FORMATTER)
    return handler


def debug(
    debug_prefix,
    product='firmware',
    clear=False,
    do_console=True,
    directory='/var/%(product)s/log',
    log_level=logging.DEBUG,
    do_jsonlog=True,
):
    """Create a debugging/development log with the given prefix"""
    if not ENABLED:
        enable(do_console=do_console, do_jsonlog=do_jsonlog)
        directory = directory % locals()
        filename = '%(directory)s/%(debug_prefix)s.log' % locals()
        handler = rotating_logfile(
            filename,
            clear=clear,
        )
        root = logging.getLogger('')
        root.setLevel(log_level)
        root.addHandler(handler)


def with_debug(
    debug_prefix, product='firmware', clear=False, do_console=True, do_jsonlog=True
):
    def with_debug_dec(base):
        @wraps(base)
        def debug_wrapped(*args, **named):
            debug(
                debug_prefix,
                product=product,
                clear=clear,
                do_console=do_console,
                do_jsonlog=do_jsonlog,
            )
            return base(*args, **named)

        return debug_wrapped

    return with_debug_dec


def with_logging(do_console=True, do_jsonlog=True):
    def with_logging_dec(base):
        @wraps(base)
        def logging_wrapped(*args, **named):
            enable(do_console=do_console, do_jsonlog=do_jsonlog)
            return base(*args, **named)

        return logging_wrapped

    return with_logging_dec


def log_on_failure(log):
    """Log unhandled exceptions to the given log"""

    def log_on_failure_dec(function):
        @wraps(function)
        def wrapper(*args, **named):
            try:
                return function(*args, **named)
            except (KeyboardInterrupt, SystemExit) as err:
                log.info("Exiting on %s", err.__class__.__name__)
                raise
            except Exception:
                log.exception('Failure running %s with %s %s', function, args, named)
                raise

        return wrapper

    return log_on_failure_dec
