import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { withRouter } from 'react-router';

import Collapse from '@material-ui/core/Collapse';
import Drawer from '@material-ui/core/Drawer';
import List from '@material-ui/core/List';
import ListItem from '@material-ui/core/ListItem';
import ListItemText from '@material-ui/core/ListItemText';
import { withStyles } from '@material-ui/core/styles';
import MuiIcon from 'MuiIcon';
import { NavLink } from 'react-router-dom';
import { getActiveSection, getActiveRoutes } from 'navigationUtils';
import {with_focus} from 'dash/focusprovider';
import WithCurrentUser from 'withcurrentuser';
import Sysinfo from 'dash/sidebarinfo';
import {USER_RESOLVED} from 'storages/ajaxobservables';

const styles = theme => ({
    drawerPaper: {
        marginTop: theme.toolbar.height,
        height: `calc(100vh - ${theme.toolbar.height})`,
        background: theme.palette.white,
        whiteSpace: 'nowrap',
        width: theme.sidebarWidth
    },
    drawerPaperClose: {
        overflowX: 'hidden',
        color: 'white',
        width: theme.spacing(0.5)
    },
    navigationList: {
        flex: 1
    },
    nestedList: {
        opacity: 0.75
    },
    productLogo: {
        width: `${theme.sidebarWidth - 20}px !important`,
        height: 40,
        marginTop: theme.spacing(2),
        marginBottom: theme.spacing(2),
        backgroundRepeat: 'no-repeat',
        backgroundSize: '80%',
        backgroundPosition: 'center',
        backgroundImage: "url('/ptheme/img/logo.svg')",
    }
});

const current_user_can_view = (item, user) => {
    if (! (item.public || user.id)) {
        return false;
    }
    if (! item.permission) {
        return true;
    }
    if (user.has_permission(item.permission)){
        return true;
    }
    // if (user && !(user.is_anonymous)) {
    //     console.log(`Cannot view ${user.name} ${item.path} ${item.permission}`);
    // }
    return false;
};


class AppSidebar extends React.Component {
    state = {
        expanded: {}
    };

    componentDidMount() {
        const { routes, location } = this.props;
        const activeSection = getActiveSection(routes, location.pathname);

        if (activeSection) {
            this.setState({
                expanded: {
                    [activeSection.header]: true
                }
            });
        }
        USER_RESOLVED.listen(this.on_user_resolved);
    }
    componentWillUnmount() {
        USER_RESOLVED.ignore(this.on_user_resolved);
    }
    on_user_resolved = (user) => {
        this.setState({});
    }

    isSelected = route => {
        const { location, routes } = this.props;
        const selected = getActiveRoutes( routes, location.pathname );
        if (selected) {
            return !! selected.find( r => r.path == route.path );
        }
        return false;
    };

    handleClick = header => {
        const { expanded } = this.state;

        this.setState({
            expanded: {
                ...expanded,
                [header]: !expanded[header]
            }
        });
    };

    renderNestedList(item) {
        const { expanded } = this.state;
        const { classes, user } = this.props;
        const isExpanded = expanded[item.header];
        if (!current_user_can_view(item, user)) {
            return null;
        }

        return (
            <div key={`${item.path}`}>
                <ListItem onClick={() => this.handleClick(item.header)}>
                    <ListItemText primary={item.header} />
                    {isExpanded ? (
                        <MuiIcon name="expand_less" />
                    ) : (
                        <MuiIcon name="expand_more" />
                    )}
                </ListItem>
                <Collapse in={isExpanded}>
                    <List className={classes.nestedList}>
                        {item.routes
                            .filter(page => !page.hideInNav)
                            .filter(page => current_user_can_view(page, user))
                            .map(page => this.renderListItem(page, true))}
                    </List>
                </Collapse>
            </div>
        );
    }

    renderListItem(page, nested = false) {
        const {focus,user} = this.props;
        if (!current_user_can_view(page, user)) {
            // console.debug(`Cannot view ${page.path} ${page.permission}`);
            return null;
        }
        const selected = this.isSelected(page);
        return (
            <ListItem
                key={`${page.path}`}
                selected={selected}
                component={NavLink}
                to={page.link_url || page.path}
                dense={!!nested}
            >
                <ListItemText primary={page.header} />
            </ListItem>
        );
    }

    render() {
        const { classes, routes, user } = this.props;

        return (
            <Drawer
                variant="permanent"
                classes={{
                    paper: classNames(classes.drawerPaper)
                }}
            >
                <List className={classes.navigationList}>
                    {routes
                        .filter(x => !x.hideInNav)
                        .filter(x => current_user_can_view(x,user))
                        .map(
                            item =>
                                item.routes && item.routes.length
                                    ? this.renderNestedList(item)
                                    : this.renderListItem(item)
                        )}
                </List>
                <div className="overall-header">
                    <Sysinfo />
                    <div
                        className={classNames(
                            'product-logo',
                            classes.productLogo
                        )}
                        style={{
                            backgroundUrl: routes.product_logo,
                        }}
                    />
                </div>
            </Drawer>
        );
    }
}

AppSidebar.propTypes = {
    classes: PropTypes.object.isRequired,
    routes: PropTypes.array.isRequired
};

export default withRouter(withStyles(styles)(with_focus(WithCurrentUser(AppSidebar))));
