import React from 'react';

import { shallow } from 'enzyme';
import sinon from 'sinon';

import { SystemConfiguration } from 'dash/SystemConfiguration';
import MuiLoader from 'MuiLoader';
import EditingCard from 'dash/editingcard';

const default_has_permission = () => true;

describe('SystemConfiguration', () => {
    let current_user_stub;
    const setup = (propOverrides = {}, has_permission=default_has_permission) => {
        const props = {
            system: {},
            forms: [
                {
                    title: 'location',
                    permission: 'config.location',
                    get_summary: sinon.spy(),
                    render_component: sinon.spy()
                },
                {
                    title: 'power',
                    permission: 'config.power',
                    get_summary: sinon.spy(),
                    render_component: sinon.spy()
                },
            ],
            location: {},
            user: {
                'has_permission': has_permission,
            },
            ...propOverrides
        };

        const wrapper = shallow(<SystemConfiguration {...props} />);

        return {
            props,
            wrapper,
            instance: wrapper.instance(),
            current_user: current_user_stub,
        };
    };

    it('renders a MuiLoader if not system', () => {
        const { wrapper } = setup({
            system: null
        });

        expect(wrapper.find(MuiLoader)).to.have.lengthOf(1);
    });

    it('does not render a form if has_permission is false', () => {
        const { wrapper, props } = setup(
            {},
            permission => {
                return permission === 'config.power';
            },
        );
        const expansionSections = wrapper.find(EditingCard);

        expect(props.forms).to.have.lengthOf(2);
        expect(expansionSections).to.have.lengthOf(1);
        expect(expansionSections.at(0).prop('section').title).to.equal('power');
    });

    it('sets defaultExpanded to true if form title matches location hash', () => {
        const { wrapper } = setup({
            location: {
                hash: '#power'
            }
        });

        const expansionSections = wrapper.find(EditingCard);
        expect(expansionSections).to.have.lengthOf(2);
        console.log(`Section is ${expansionSections.at(0).prop('section').debug}`);
        expect(expansionSections.at(0).prop('section')).not.to.equal(undefined);
        expect(expansionSections.at(0).prop('section').title).to.equal('location');
        expect(expansionSections.at(0).prop('defaultExpanded')).to.equal(false);
        expect(expansionSections.at(1).prop('section').title).to.equal('power');
        expect(expansionSections.at(1).prop('defaultExpanded')).to.equal(true);
    });

    it('calls form.render_component for each form', () => {
        const { props } = setup();

        props.forms.forEach(form => {
            expect(form.render_component).to.have.been.called;
        });
    });
});
