import React from 'react';

import { shallow, render } from 'enzyme';
import sinon from 'sinon';

import { SystemPowerActions } from 'dash/SystemPowerActions';
import * as reactform from 'reactform/reactform';
import MuiSnackbar from 'MuiSnackbar';

describe('SystemPowerActions', () => {
    let form_action, construct_controls;

    const setup = (propOverrides = {}) => {
        const props = {
            target: { type: 'system' },
            classes: {},
            ...propOverrides,
        };

        const wrapper = shallow(<SystemPowerActions {...props} />);

        return {
            props,
            wrapper,
            instance: wrapper.instance(),
        };
    };

    before(() => {
        form_action = sinon
            .stub(reactform, 'form_action')
            .returns({ render: () => '' });
        construct_controls = sinon.spy(reactform, 'construct_controls');
    });

    afterEach(() => {
        form_action.resetHistory();
        construct_controls.resetHistory();
    });

    after(() => {
        form_action.restore();
        construct_controls.restore();
    });

    it('renders a ReactForm', () => {
        const { wrapper } = setup();
        expect(wrapper.find(reactform.ReactForm)).to.have.lengthOf(1);
    });

    it('does not render a MuiSnackbar if state.response is falsey', () => {
        const { wrapper } = setup();
        expect(wrapper.find(MuiSnackbar)).to.have.lengthOf(0);
    });

    it('renders a MuiSnackbar if state.response exists', () => {
        const { wrapper } = setup();
        const response = {
            success: true,
            message: 'some message',
        };
        wrapper.setState({ response });

        const snackbar = wrapper.find(MuiSnackbar);
        expect(snackbar).to.have.lengthOf(1);
        expect(snackbar.props().variant).to.equal('success');
        expect(snackbar.props().onClose).to.equal(wrapper.instance().onClose);
    });

    it('getFormActions calls form_action with actions', () => {
        const { instance } = setup();

        instance.getFormActions();

        expect(form_action.callCount).to.equal(4);

        instance.actions.forEach(action => {
            expect(form_action).to.have.been.calledWith({
                ...action,
                help_text: action.help_text,
                callback: instance.handleResponse,
            });
        });
    });

    it('getFormActions calls construct_controls', () => {
        const { instance } = setup();
        instance.getFormActions();
        expect(construct_controls.callCount).to.equal(1);
    });

    it('handleResponse sets state.response', () => {
        const { wrapper, instance } = setup();
        const response = { success: true };
        instance.handleResponse(response);
        expect(wrapper.state('response')).to.deep.equal(response);
    });

    it('MuiSnackbar.onClose sets state.response to null', () => {
        const { wrapper, instance } = setup();
        const response = { success: true };
        wrapper.setState({ response });
        wrapper.find(MuiSnackbar).prop('onClose')();
        expect(wrapper.state('response')).to.equal(null);
    });
});
