import React from 'react';
import classNames from 'classnames';
import { withStyles } from '@material-ui/core/styles';

import generic_filter from 'reactform/genericfilter';
import concrete_columns from './concretecolumns';

import theme from 'dash/theme';
import { useTable, useSortBy, useFilters } from 'react-table';
import useGUIState from 'dash/guistate';
import CssBaseline from '@material-ui/core/CssBaseline';
import Table from '@material-ui/core/Table';
import TableBody from '@material-ui/core/TableBody';
import TableCell from '@material-ui/core/TableCell';
import TableHead from '@material-ui/core/TableHead';
import TableRow from '@material-ui/core/TableRow';
import TextField from '@material-ui/core/TextField';
import PreferenceMenu from './atxtablepreferencemenu';
import "./atx-react-table.css";

import IconButton from '@material-ui/core/IconButton';
import MoreVert from '@material-ui/icons/MoreVert';
import Menu from '@material-ui/core/Menu';
import MenuItem from '@material-ui/core/MenuItem';
import RadioButtonChecked from '@material-ui/icons/RadioButtonChecked';
import RadioButtonUnchecked from '@material-ui/icons/RadioButtonUnchecked';

const styles = (theme) => ({
    clickableRow: {
        '&:hover': {
            backgroundColor: theme.palette.grey[200],
            cursor: 'pointer',
        },
    },
    relativeHolder: {
        position: 'relative',
    },
    header: {
        backgroundColor: theme.palette.headers.table,
    },
    headerGroup: {
        backgroundColor: theme.palette.headers.tableGroup,
    },
    sortIndicator: {
        position: 'absolute',
        top: '-9px',
        right: '-3px',
        color: 'green',
        fontSize: 'smaller',
    },
    filterField: {
        fontSize: '.75em',
        color: theme.palette.grey[400],
    },
    firstField: {
        borderLeft: 'thin solid #e0e0e0',
    },
    quick_setting_trigger: {
        position: 'absolute',
        bottom: 0,
        left: 0,
        right: 0,
    },
    quickSettingTh: {
        textAlign: 'center',
        backgroundColor: theme.palette.headers.table,
        padding:0,
        margin:0,
        '& .edit-intent a': {
            display: 'block',

        }
    },
    defaultAlignment: {
        alignContent: 'start',
        justifyContent: 'center',
    },
    root: {
        '&.ReactTable .rt-resizer': {
            width: 12,
            right: -6,
        },
    },
});


function DefaultColumnFilter({
    column: { filterValue, preFilteredRows, setFilter },
}) {
    const count = preFilteredRows.length;

    return (
        <TextField
            value={filterValue || ''}
            onChange={e => {
                setFilter(e.target.value || undefined); // Set undefined to remove the filter entirely
            }}
            size='small'
            onClick={e => e.stopPropagation()}
        />
    );
}



const RenderATXTable = withStyles(styles)(useGUIState(React.forwardRef((props,ref) => {
    const {columns,data,classes,onFilterRows,...childProps} = props;
    const [current,setState] = React.useState(null);
    const preference_key = props.preference_key;
    let initialGUIState={},updateGUIState;
    if (props.useGUIState) {
        [initialGUIState,updateGUIState] = props.useGUIState();
        // console.log(`Initial GUI state ${JSON.stringify(initialGUIState)}`);
    }
    const initialState = initialGUIState || {
    };
    if (props.defaultSorting && initialState.sortBy === undefined) {
        initialState.sortBy = props.defaultSorting;
    }
    const {
        getTableProps,
        getTableBodyProps,
        headerGroups,
        allColumns,
        rows,
        prepareRow,
    } = useTable(
        {
            columns,
            data,
            initialState: initialState,
            autoResetPage: false,
            autoResetExpanded: false,
            autoResetGroupBy: false,
            autoResetSelectedRows: false,
            autoResetSortBy: false,
            autoResetFilters: false,
            autoResetRowState: false,
            useControlledState: (state) => {
                return {
                    ...state,
                    hiddenColumns: (initialState && initialState.hiddenColumns) || [],
                };
            },
        },
        useFilters,
        useSortBy,
    );
    const try_render_cell = (cell) => {
        try {
            return cell.render('Cell',cell);
        } catch (e) {
            return `Error rendering ${cell}: ${e}`;
        }
    };
    /* Yech, but we need to know this from outside */
    onFilterRows(rows);
    return <div className={classNames(classes.relativeHolder,'prefs-holder')}>
        <Table {...getTableProps()} ref={ref} className='ReactTable atx-table' >
            <TableHead className={classNames(classes.root,'rt-thead','-headerGroups')}>
                {headerGroups.map((headerGroup,headerGroupIndex) => {
                    const have_filters = headerGroup.headers.filter(column => column.canFilter).length;
                    const are_sortable = headerGroup.headers.filter(column => (!column.columns)).length;
                    const have_quicksettings = headerGroup.headers.filter(column => (!!column.QuickSetting)).length;
                    const row_props = headerGroup.getHeaderGroupProps();
                    const rows = [];
                    const names = <TableRow
                        {...row_props}
                        className={classNames(
                            are_sortable?classes.header:classes.headerGroup,
                            'rt-tr',
                            are_sortable?'names':'groups',
                            are_sortable?classes.clickableRow:null,
                            row_props.className,
                        )}
                        key={`rt-thead-${headerGroupIndex}`}
                    >
                        {headerGroup.headers.map(column => {
                            const can_sort = (!column.columns) && !(column.sortable === false);
                            const cell_props = column.getHeaderProps(column.getSortByToggleProps());
                            return <TableCell
                                {...cell_props}
                                className={
                                    classNames(
                                        classes.headerCell,
                                        'rt-th',
                                        classes.relativeHolder,
                                        cell_props.className,
                                        column.size?`col-size-${column.size}`: null,
                                    )
                                }
                            >
                                {column.render('Header')}
                                {
                                    can_sort &&
                                    <div className={classes.sortIndicator}>
                                        {column.isSorted
                                            ? column.isSortedDesc
                                                ? '▼'
                                                : '▲'
                                            : ''}
                                    </div>
                                }
                            </TableCell>;
                        })}
                    </TableRow>;
                    rows.push(names);
                    if (have_quicksettings) {
                        const qs_row_props = headerGroup.getHeaderGroupProps();
                        rows.push(
                            <TableRow
                                {...qs_row_props}
                                className={classNames(
                                    'rt-tr',
                                    'quicksettings',
                                )}
                                key='quicksettings-${headerGroupIndex}'
                            >
                                {headerGroup.headers.map(column => {
                                    return <TableCell
                                        {...column.getHeaderProps()}
                                        className={classNames(
                                            classes.header,
                                            'rt-th',
                                            'quicksetting',
                                            classes.quickSettingTh,
                                            column.size?`col-size-${column.size}`: null,
                                        )}
                                    >
                                        {
                                            (column.QuickSetting) &&
                                                column.render('QuickSetting')
                                        }
                                    </TableCell>;
                                })}
                            </TableRow>
                        );
                    }
                    if (have_filters) {
                        const filter_row_props = headerGroup.getHeaderGroupProps();

                        rows.push(
                            <TableRow
                                {...filter_row_props}
                                className='rt-tr filters' key='filters-${headerGroupIndex}'
                            >
                                {headerGroup.headers.map(column => {
                                    return <TableCell
                                        {...column.getHeaderProps()}
                                        className={classNames(
                                            classes.header,
                                            'rt-th',
                                            column.size?`col-size-${column.size}`: null,
                                        )}>
                                        {
                                            (column.canFilter) &&
                                                column.render('Filter',column)
                                        }
                                    </TableCell>;
                                })}
                            </TableRow>
                        );
                    }
                    return rows;
                })}

            </TableHead>
            <TableBody {...getTableBodyProps()}>
                {rows.map((row, i) => {
                    prepareRow(row);
                    const row_props = row.getRowProps();
                    return (
                        <TableRow
                            {...row_props}
                            className={classNames(
                                row_props.className,
                                'rt-tr',
                                i%2?'-even':'-odd'
                            )}
                            key={`body-r-${i}`}
                        >
                            {row.cells.map(cell => {
                                const custom_props = (props.getTdProps && props.getTdProps(
                                    null,
                                    cell.row,
                                    cell.column,
                                    cell.original,
                                )) || {};
                                return (
                                    <TableCell
                                        {...cell.getCellProps()}
                                        {...custom_props}
                                        className={classNames(
                                            custom_props.className,
                                            'rt-td',
                                            cell.column.size?`col-size-${cell.column.size}`: null,
                                            `col-${cell.column.id}`,
                                            cell.column.form_field?`field-${cell.column.form_field.name}`:null,
                                        )}
                                    >
                                        {try_render_cell(cell)}
                                    </TableCell>
                                );
                            })}
                        </TableRow>
                    );
                })}
            </TableBody>
        </Table>
        {
            preference_key && <PreferenceMenu
                allColumns={allColumns}
                updateGUIState={updateGUIState}
                initialState={initialState}
                hideColumnCallback={(newGUIState) => {
                    setState({});
                }}
            />
        }
    </div>;
})));
RenderATXTable.displayName = 'RenderATXTable';


class BaseATXTable extends React.Component {
    static defaultProps = {
        pageSize: 500,
        showPagination: false,
        minRows: 0,
        filterable: true,
        defaultFilterMethod: generic_filter,
        collapseOnDataChange: false,
        defaultCell: null,
        keyField: '__pk__',
    }
    rows = [];
    getFinalColumns = () => {
        const { defaultCell, columns } = this.props;
        if (defaultCell && columns) {
            concrete_columns(columns).map(column => {
                if (!column.Cell) {
                    column.Cell = defaultCell;
                }
            });
        }
        concrete_columns(columns).map(column => {
            if (!column.Filter) {
                column.Filter = DefaultColumnFilter;
            }
        });
        /* Unique column ids are a requirement in react-table 7.x */
        const seen = {};
        concrete_columns(columns).map(column => {
            const original = column.id;
            if (column.id) {
                let count = 1;
                while (seen[column.id]) {
                    count += 1;
                    column.id = `${original}-${count}`;
                }
                seen[column.id] = column;
            }
        });
        return columns;
    }
    overRideFunctions = (props) => {
        /* Create functions that do overrides for our common bits */
        const {
            data,
            ...final_props
        } = props;

        ['getTheadTrProps', 'getTheadFilterProps', 'getTrGroupProps'].map(key => {
            if (props[key]) {
                final_props[key] = (state, rowInfo) => {
                    // console.log(`Callback on ${key} with child def`);
                    return this[key](state, rowInfo, props[key](state, rowInfo));
                };
            } else {
                final_props[key] = (state, rowInfo) => {
                    // console.log(`Callback on ${key} with no child def`);
                    return this[key](state, rowInfo, {});
                };
            }
        });
        if (data && data.length > final_props.pageSize) {
            final_props.showPagination = true;
        }
        return {
            // getTheadTrProps: this.getTheadTrProps,
            // getTheadFilterProps: this.getTheadFilterProps,
            // getTrGroupProps: this.getTrGroupProps,
            onFilterRows: this.onFilterRows,
            getTheadGroupProps: this.getTheadGroupProps,
            getTrProps: this.getTrProps,
            getTdProps: this.props.getTdProps,
            data,
            ...final_props,
        };
    }
    getTheadGroupProps(state, rowInfo, column, instance) {
        const { classes } = this.props;
        return {
            className: classNames(classes.header),
        };
    }
    getTheadTrProps = (state, rowInfo, baseprops) => {
        /* add standard header properties for ATXTable */
        const { classes } = this.props;
        return {
            ...baseprops,
            className: classNames((baseprops && baseprops.className), classes.header),
        };
    }
    getTheadFilterProps = (state, rowInfo, baseprops) => {
        const { classes } = this.props;
        return {
            ...baseprops,
            className: classNames(baseprops.className, classes.header),
        };
    }
    getTrProps = (state, rowInfo, baseProps) => {
        const target = rowInfo.original;
        const key = target[this.props.keyField];
        return {
            key: key,
            ...baseProps,
        };
    }
    getTrGroupProps = (state, rowInfo, baseProps) => {
        /* Use keyField to set stable keys on row */
        const target = rowInfo.original;
        return {
            key: target[this.props.keyField] || rowInfo.index,
        };
    }
    onFilterRows = (rows) => {
        /* Callback to get the reference to the underlying react data-table */
        this.rows = rows;
        if (this.props.onFilterRows) {
            this.props.onFilterRows(rows);
        }
    }
    render() {
        const props = this.overRideFunctions(this.props);
        const columns = this.getFinalColumns();
        const final_props = {
            ...props,
            columns: columns,
            className: classNames(
                props.className,
                'atx-table',
                this.props.classes.root
            ),
        };
        // {...final_props}
        const ref_func = this.props.tableRef || this.table_ref_callback;

        return <RenderATXTable
            ref={ref_func}
            {...final_props}
        />;
    }
}
const ATXTable = withStyles(styles)(BaseATXTable);
ATXTable.displayName = 'ATXTable';

class SelectingATXTable extends React.Component {
    static defaultProps = {
        keyField: '__pk__',
    }
    state = {
        selected: {},
    }
    on_row_click = (record) => {
        const key = record[this.props.keyField];
        const selected = {
            ...this.state.selected,
        };
        if (selected[key]) {
            delete selected[key];
        } else {
            selected[key] = record;
        }
        this.setState({
            selected: selected,
        });
        if (this.props.on_change) {
            this.props.on_change(selected);
        }
    }
    // <IconButton title="Select all" onClick={(evt) => {
    //     const new_selections = {};
    //     const instance = this.get_current_vct();
    //     instance.vct.map((channel,i)=>{
    //         new_selections[i] = channel;
    //     });
    //     this.setState({'vct_selections': new_selections});
    // }}><Icon>select_all</Icon></IconButton>
    // <IconButton title="Clear all" onClick={(evt) => {
    //     this.setState({'vct_selections':{}});
    // }}><Icon>clear_all</Icon></IconButton>

    getTrProps = (state, rowInfo, baseProps) => {
        const { classes } = this.props;
        const target = rowInfo.original;
        const key = target[this.props.keyField];
        const base = (
            this.props.getTrProps && this.props.getTrProps(state, rowInfo, baseProps)
        ) || (
            {}
        );
        const selected = this.state.selected[key] ? 'selected' : null;
        return {
            key: key,
            ...base,
            className: classNames(base.className, selected),
            onClick: () => this.on_row_click(target),
        };
    }
    render() {
        return <ATXTable {...this.props} getTrProps={this.getTrProps} />;
    }
}

export default ATXTable;
export { ATXTable, SelectingATXTable };
