/* Data processing operations for pulling out graph series */
import {parallel_sets, apply_accessor} from './accessor';
import order_by_keys from './orderbykeys';
import {COLORS} from './colors';
import PropTypes from 'prop-types';

function filter_exclude(records, excluder) {
    /* Filter accepted/unaccepted for records */
    if (!excluder) {
        return [records,[]];
    }
    const result = [[],[]];
    records.map((record)=> {
        if (excluder(record)) {
            result[1].push(record);
        } else {
            result[0].push(record);
        }
    });
    return result;
}

function group_by_accessor(records, metric) {
    /* Group records in series by metric.accessor */
    if ((!records) || (!records.map)) {
        return [];
    }
    const groups = {};
    // eslint-disable-next-line no-inner-declarations
    function add_to_group(__key__,record) {
        if (!groups[__key__]) {
            groups[__key__] = [];
        }
        groups[`${__key__}`].push(record);
    }
    records.map(rec => {
        var __key__ = apply_accessor(rec, metric.accessor);
        if (__key__ === undefined) {
            __key__ = null;
        }
        add_to_group(__key__,rec);
    });
    return Object.keys(groups).map((__key__) => {
        const label = (metric.labels && metric.labels[__key__]) || __key__;
        const value = groups[__key__];
        const record =  {
            '__key__': __key__,
            'label': label,
            'value': value,
        };
        const color = (__key__ === '__excluded__') ? 
            COLORS.DISABLED: 
            ((metric.colors && metric.colors[__key__]) || undefined);
        if (color !== undefined && color !== null) {
            record.color = color;

        }
        return record;
    });
}
function enumeration_order(series,metric) {
    if (metric.order) {
        // console.info(`Applying order to ${metric.__key__}`);
        return order_by_keys(metric.order, series, '__key__');
    }
    return series;
}
function aggregate_series(series,format) {
    if (format.aggregate) {
        return series.map((record) => {
            return format.aggregate(record);
        });
    }
    return series;
}

function extract_series_grouped(__key__,partial,metric, format) {
    let [records,excluded] = filter_exclude(partial, metric.exclude);
    // console.log(`${records.length} records to group`);
    const series = group_by_accessor(records, metric);
    if (!('ignore_excluded' in metric) && excluded.length) {
        const label = (metric.labels && metric.labels['__excluded__']) || 'Excluded';
        series.push({
            '__key__': __key__? `${__key__}.__excluded__`: '__excluded__',
            'label': label,
            'value': excluded,
            // 'color': COLORS.DISABLED,
        });
    }
    // console.log(`${series.length} groups created`);
    const ordered = enumeration_order(series,metric);
    ordered.map((record,index) => {
        if (record.coordinate === undefined) {
            record.coordinate = index;
        }
        if (record.id === undefined) {
            record.id = index;
        }
    });
    return aggregate_series(ordered,format);
}
function extract_series_ungrouped(__key__,partial,metric, format) {
    if (partial === null) {
        return null;
    } else if (!partial.map) {
        console.info(`Non array in extract series ungrouped ${JSON.stringify(metric)}: ${partial}`);
        return null;
    }
    return partial.map((record,index) => {
        if (metric.exclude && metric.exclude(record)) {
            return;
        }
        const value = apply_accessor(record, metric.accessor);
        const label = (metric.label && metric.label(record)) || index;
        return {
            '__key__': __key__,
            'id': (metric.id && metric.id(record)) || index,
            'label': label,
            'value': value,
            'coordinate': (metric.coordinate? metric.coordinate(record): index),
        };
    });
}

function metric_dataset(dataset, metric) {
    /* Get the overall dataset related to the metric */
    const parallel = parallel_sets(
        dataset,
        metric.dataset_accessor || metric.__key__.split('.')[0],
    );
    // console.info(`Parallel sets: ${JSON.stringify(Object.entries(parallel))} extracted for ${metric.__key__}`);
    return parallel;
}

function extract_data(graph, dataset, metric, format) {
    /* Create graph-compatible dataset 
    
    returns [{__key__:string,value:any,label:string,id:any},...]
    */
    // if ((!dataset) || (!dataset.map)) {
    //     console.info("Null dataset passed to extract_data");
    //     return [];
    // }
    const parallel = metric_dataset(dataset, metric);

    const series = Object.entries(parallel).map((r,index) => {
        let [__key__,partial] = r;
        if (format.group_by) {
            // console.info("Grouped series");
            return extract_series_grouped(__key__,partial,metric, format);
        } else {
            /* Un-grouped data, so the result will be one record per source record */
            // console.debug("Un-grouped series");
            return extract_series_ungrouped(__key__, partial, metric, format);
        }
    });
    return series;
}
const AccessorType = PropTypes.oneOfType([PropTypes.string,PropTypes.func]);

const MetricType = PropTypes.shape({
    dataset_accessor: AccessorType,
    __key__: PropTypes.string.isRequired,
});
const FormatType = PropTypes.shape({
    group_by: PropTypes.func,
    aggregate: PropTypes.func,
});
metric_dataset.propTypes = {
    'dataset': PropTypes.array.isRequired,
    'metric': MetricType.isRequired,
};
extract_data.propTypes = {
    'metric': MetricType.isRequired,
    'dataset': PropTypes.array.isRequired,
    'format': FormatType.isRequired,
};

export default extract_data;
export {
    extract_data, 
    extract_series_grouped, 
    extract_series_ungrouped,
    filter_exclude,
    group_by_accessor,
    metric_dataset,
    enumeration_order,
};